from os import PRIO_PGRP
from typing import Optional, List
from enum import Enum
import re


# ### GRAMMAR ###
# Filter -> Or#
# Or -> Or,Add|Add
# Add -> Ad+Exp'|Exp'
# Exp' -> Exp|(Or)
# Exp -> ParamSepVal
# Param -> String
# V -> null|true|false|Number|Literal|String|RelDate
# RelDate -> now(-|+)Number(d|w|M|y|h|m|s)
# Sep -> :|:Op
# Op -> -|>|>=|<=|~|~^|~$|[{Val}]|-[{Val}]


__all__ = ["Operator", "FilterObject", "Query"]


class Operator(str, Enum):
    NOT = "-"
    GREATER = ">"
    GREATER_EQUAL = ">="
    LESS = "<"
    LESS_EQUAL = "<="
    CONTAINS = "~"
    STARTS_WITH = "~^"
    ENDS_WITH = "~$"


class FilterObject:
    def resolve(self) -> str:
        return ""

    def __str__(self) -> str:
        return self.resolve()


class ParenthesizedFilter(FilterObject):
    def __init__(self, operand: FilterObject):
        self.operand = operand

    def resolve(self) -> str:
        return "(" + self.operand.resolve() + ")"


class OrFilter(FilterObject):
    def __init__(self, operands: List[FilterObject]):
        self.operands = operands

    def resolve(self) -> str:
        return ",".join(o.resolve() for o in self.operands)

    def __add__(self, other: FilterObject):
        if isinstance(other, OrFilter):
            return OrFilter(self.operands + other.operands)
        return OrFilter(self.operands + [other])

    def __mul__(self, other: FilterObject):
        if isinstance(other, OrFilter):
            return AndFilter([ParenthesizedFilter(self), ParenthesizedFilter(other)])
        return AndFilter([ParenthesizedFilter(self), other])


class AndFilter(FilterObject):
    def __init__(self, operands: List[FilterObject]):
        self.operands = operands

    def resolve(self) -> str:
        return "+".join(o.resolve() for o in self.operands)

    def __add__(self, other: FilterObject) -> OrFilter:
        return OrFilter([self, other])

    def __mul__(self, other: FilterObject):
        if isinstance(other, AndFilter):
            return AndFilter(self.operands + other.operands)
        if isinstance(other, OrFilter):
            return AndFilter(self.operands + [ParenthesizedFilter(other)])
        if isinstance(other, Query):
            return AndFilter(self.operands + [other])


class Query(FilterObject):
    def __init__(self, property: str, value, operation: Optional[Operator] = None):
        if not re.match(r"^[a-zA-Z_][\w\.]*$", property):
            raise Exception(f"Property {property} has invalid format")

        self.property = property.lower()
        self.value = value
        self.operation = operation

    def resolve(self) -> str:
        def resolve_value(value) -> str:
            if value is None:
                return "null"

            if isinstance(value, bool):
                return str(value).lower()

            if isinstance(value, str):
                if re.match(r"^now[\+\-][\d]+[dwMyhms]$", value):
                    return value
                return "'" + value.replace("'", "\\'").replace('"', '\\"') + "'"

            return str(value)

        if self.operation:
            return f"{self.property}:{self.operation.value}{resolve_value(self.value)}"
        return f"{self.property}:{resolve_value(self.value)}"

    def __mul__(self, other: FilterObject) -> AndFilter:
        if isinstance(other, OrFilter):
            return AndFilter([self, ParenthesizedFilter(other)])
        if isinstance(other, AndFilter):
            return AndFilter([self] + other.operands)
        return AndFilter([self, other])

    def __add__(self, other: FilterObject) -> OrFilter:
        if isinstance(other, OrFilter):
            return OrFilter([self] + other.operands)
        return OrFilter([self, other])
