from ghost.client import AdminAPIClient
from ghost.exceptions import BadRequestException, GhostException, NotFoundException
from ghost.models import BaseModel
from datetime import datetime
from typing import Optional


class Label(BaseModel):
    printable = ["id", "name", "slug"]

    def __init__(
        self,
        client: AdminAPIClient,
        id: Optional[str] = None,
        name: Optional[str] = None,
        slug: Optional[str] = None,
        created_at: Optional[datetime] = None,
        updated_at: Optional[datetime] = None,
    ):
        super().__init__()

        if not id and not slug:
            raise GhostException("One of `id` or `slug` must be passed")

        self.client: AdminAPIClient = client
        self.id: str | None = id
        self.name: str | None = name
        self.slug: str | None = slug
        self.created_at: datetime | None = created_at
        self.updated_at: datetime | None = updated_at

    def get(self):
        try:
            response = self.client.get("/labels/", params={"filter": f"id:{self.id}"})
            if not response or not response["labels"] or not len(response["labels"]):
                raise BadRequestException("/labels/", 404, "")
        except BadRequestException as e:
            if e.status_code != 404:
                raise
            response = self.client.get(
                "/labels/", params={"filter": f"slug:{self.slug}"}
            )

        if not response or not response["labels"] or not len(response["labels"]):
            raise NotFoundException(self, {"id": self.id, "slug": self.slug})

        self._assign_from_data(response["labels"][0])
        super().get()
        return self

    def create(self):
        super().create()

        if not self.slug and not self.name:
            raise GhostException(
                "One of 'name' or 'slug' must be non-null to create Label."
            )

        label_data = self.to_dict()

        response = self.client.post("/labels/", payload={"labels": [label_data]})
        self._assign_from_data(response["labels"][0])
        self.exists = True

        return self

    def update(self):
        super().update()

        label_data = self.to_dict()

        response = self.client.put(
            f"/labels/{self.id}", payload={"labels": [label_data]}
        )

        if not response or not response["labels"] or not len(response["labels"]):
            raise GhostException("Could not update label: empty answer from server.")

        self._assign_from_data(response["labels"][0])

        return self

    def to_dict(self) -> dict:
        return {"id": self.id, "name": self.name, "slug": self.slug}

    def summarize(self) -> dict:
        return {"id": self.id, "slug": self.slug}

    def _assign_from_data(self, data: dict):
        self.id = data.get("id")
        self.name = data.get("name")
        self.slug = data.get("slug")
        self.created_at = datetime.fromisoformat(str(data.get("created_at")))
        self.updated_at = datetime.fromisoformat(str(data.get("updated_at")))

    def __hash__(self) -> int:
        return hash((self.id, self.slug))


if __name__ == "__main__":
    from dotenv import load_dotenv
    from os import getenv

    load_dotenv()

    client = AdminAPIClient(getenv("GHOST_API_TOKEN", ""), getenv("GHOST_URL", ""))
    label = Label(client, "63e9160fca774efb785b279f").get()
    print(label)
