from ghost.models import BaseModel, Label, Newsletter, Tier
from ghost.client import AdminAPIClient
from ghost.exceptions import BadRequestException, GhostException, NotFoundException
from ghost.filter import FilterObject
from typing import Iterable
from datetime import datetime
from rich.progress import (
    BarColumn,
    Progress,
    MofNCompleteColumn,
    TextColumn,
    TaskProgressColumn,
    TimeElapsedColumn,
)

from ghost.models.tier import TierType


class Member(BaseModel):
    printable = ["id", "email", "created_at", "note"]

    def __init__(
        self,
        client: AdminAPIClient,
        id: str | None = None,
        name: str | None = None,
        email: str | None = None,
        labels: set[Label] = set(),
        newsletters: set[Newsletter] = set(),
        tiers: set[Tier] = set(),
        note: str | None = None,
        avatar_image: str | None = None,
        email_suppression: bool = False,
    ):
        super().__init__()

        if not id and not email:
            raise GhostException("One of `id` or `email` must be passed")

        self.client = client
        self.id = id
        self.name = name
        self.email = email
        self.labels = labels
        self.newsletters = newsletters
        self.tiers = tiers
        self.note = note
        self.avatar_image = avatar_image
        self.email_suppression = email_suppression

    def get(self):
        if self.id is None and self.email is None:
            raise GhostException("Cannot retrieve member with no email or id.")

        try:
            response = self.client.get("/members/", params={"filter": f"id:{self.id}"})
            if not response or not response["members"]:
                raise BadRequestException("/members/", 404, "")
        except BadRequestException as e:
            if e.status_code == 404:
                response = self.client.get(
                    "/members/", params={"filter": f"email:'{self.email}'"}
                )
            else:
                raise

        if not response["members"]:
            raise NotFoundException(self, attr={"id": self.id, "email": self.email})

        super().get()
        return self._assign_from_data(response["members"][0])

    def create(self):
        super().create()

        member_data = {
            "email": self.email,
            "avatar_image": self.avatar_image,
            "labels": [lab.summarize() for lab in self.labels],
            "newsletters": [n.summarize() for n in self.newsletters],
            "note": self.note if self.note else "",
        }

        response = self.client.post("/members/", payload={"members": [member_data]})
        self.exists = True
        return self._assign_from_data(response["members"][0])

    def update(self):
        super().update()

        member_data = dict()
        if self.email:
            member_data["email"] = self.email
        if self.avatar_image:
            member_data["avatar_image"] = self.avatar_image
        if self.labels is not None:
            member_data["labels"] = [lab.summarize() for lab in self.labels]
        if self.newsletters is not None:
            member_data["newsletters"] = [n.summarize() for n in self.newsletters]
        if self.note is not None:
            member_data["note"] = self.note

        response = self.client.put(
            f"/members/{self.id}", payload={"members": [member_data]}
        )
        return self._assign_from_data(response["members"][0])

    def add_newsletters(self, newsletter_ids: str | Iterable[str]):
        super().update()

        if isinstance(newsletter_ids, str):
            news_list = set([newsletter_ids])
        elif isinstance(newsletter_ids, set):
            news_list = newsletter_ids
        elif isinstance(newsletter_ids, Iterable):
            news_list = set(newsletter_ids)
        else:
            raise GhostException(
                f"'newsletters' needs to be str, list or set, not {type(newsletter_ids)}"
            )

        curr_news = set(n.id for n in self.newsletters)
        news_final = curr_news.union(news_list)

        payload = {"members": [{"newsletters": [{"id": n} for n in news_final]}]}
        response = self.client.put(f"/members/{self.id}", payload=payload)
        return self._assign_from_data(response["members"][0])

    def remove_newsletters(self, newsletter_ids: str | Iterable[str]):
        super().update()

        if isinstance(newsletter_ids, str):
            news_list = set([newsletter_ids])
        elif isinstance(newsletter_ids, set):
            news_list = newsletter_ids
        elif isinstance(newsletter_ids, Iterable):
            news_list = set(newsletter_ids)
        else:
            raise GhostException(
                f"'newsletters' needs to be str, list or set, not {type(newsletter_ids)}"
            )

        curr_news = set(n.id for n in self.newsletters)
        news_final = curr_news - news_list

        payload = {"members": [{"newsletters": [{"id": n} for n in news_final]}]}
        response = self.client.put(f"/members/{self.id}", payload=payload)
        return self._assign_from_data(response["members"][0])

    def add_tiers(self, tier_ids: str | Iterable[str]):
        super().update()

        if isinstance(tier_ids, str):
            tier_list = set([tier_ids])
        elif isinstance(tier_ids, set):
            tier_list = tier_ids
        elif isinstance(tier_ids, Iterable):
            tier_list = set(tier_ids)
        else:
            raise GhostException(
                f"'tier_ids' needs to be str, list or set, not {type(tier_ids)}"
            )

        curr_tiers = set(t.id for t in self.tiers)
        tiers_final = curr_tiers.union(tier_list)

        payload = {"members": [{"tiers": [{"id": t} for t in tiers_final]}]}
        response = self.client.put(f"/members/{self.id}", payload=payload)
        return self._assign_from_data(response["members"][0])

    def remove_tiers(self, tier_ids: str | Iterable[str]):
        super().update()

        if isinstance(tier_ids, str):
            tier_list = set([tier_ids])
        elif isinstance(tier_ids, set):
            tier_list = tier_ids
        elif isinstance(tier_ids, Iterable):
            tier_list = set(tier_ids)
        else:
            raise GhostException(
                f"'tier_ids' needs to be str, list or set, not {type(tier_ids)}"
            )

        curr_tiers = set(t.id for t in self.tiers)
        tiers_final = curr_tiers - tier_list

        payload = {"members": [{"tiers": [{"id": t} for t in tiers_final]}]}
        response = self.client.put(f"/members/{self.id}", payload=payload)
        return self._assign_from_data(response["members"][0])

    def send_magic_link(self):
        if not self.exists:
            raise GhostException("Cannot send magic link to member that doesn't exist")

        payload = {
            "email": self.email,
            "emailType": "signin",
            "integrityToken": self.client.api_token,
        }

        self.client.post_in_api("/members/api", "/send-magic-link", payload=payload)

        return self

    @classmethod
    def from_data(cls, client: AdminAPIClient, data: dict):
        if not data.get("member") or not data["member"].get("current"):
            raise GhostException("Cannot assign from incomplete data")

        current_member = data["member"]["current"]
        id = current_member.get("id")
        email = current_member.get("email")

        member = Member(client, id=id, email=email).get()

        return member

    @classmethod
    def all(
        cls,
        client: AdminAPIClient,
        pagesize: int | None = None,
        filters: dict | FilterObject | None = None,
        limit: int | None = None,
        progress_bar: bool = False,
    ) -> Iterable["Member"]:
        fetched: int = 0

        filter_str = ""
        if filters:
            if isinstance(filters, FilterObject):
                filter_str = filters.resolve()
            else:
                filter_str = "+".join([f"{k}:{v}" for k, v in filters.items()])

        params = dict()
        if filter_str != "":
            params["filter"] = filter_str

        if pagesize is not None:
            params["limit"] = pagesize
        else:
            params["limit"] = "all"

        data = client.get("/members/", params=params)
        fetched += len(data["members"])

        def new_member(data: dict) -> Member:
            member = Member(client, data.get("id"))._assign_from_data(data)
            member.exists = True
            return member

        yield from (new_member(m) for m in data["members"])

        def loop_condition(next) -> bool:
            if limit:
                return bool(next) and fetched < limit
            return bool(next)

        def yield_members(data):
            next = data["meta"]["pagination"]["next"]
            while loop_condition(next):
                params["page"] = next
                data = client.get("/members/", params=params)
                yield from (new_member(m) for m in data["members"])
                next = data["meta"]["pagination"]["next"]

        if progress_bar:
            with Progress(
                TextColumn("[progress.description]{task.description}"),
                BarColumn(),
                MofNCompleteColumn(),
                TaskProgressColumn(),
                TimeElapsedColumn(),
            ) as p:
                t = p.add_task(
                    f"Retrieving members from {client.base_url}",
                    total=int(data["meta"]["pagination"]["total"]),
                )
                for chunk in yield_members(data):
                    p.update(t, advance=1)
                    yield chunk
        else:
            yield from yield_members(data)

    def _assign_from_data(self, data: dict):
        self.id = data.get("id")
        self.email = data.get("email")
        self.name = data.get("name")
        self.avatar_image = data.get("avatar_image")
        self.note = data.get("note")
        self.status = data.get("status")
        self.created_at = datetime.fromisoformat(str(data.get("created_at")))
        self.updated_at = datetime.fromisoformat(str(data.get("updated_at")))
        self.email_suppression = data.get(
            "email_suppression", {"suppressed": False}
        ).get("suppressed")
        self.labels = self._get_labels(data)
        self.newsletters = self._get_newsletters(data)
        try:
            self.tiers = self._get_tiers(data)
        except KeyError:
            self.tiers = set()

        return self

    def _get_newsletters(self, data: dict):
        newsletters_data = data.get("newsletters")
        if not newsletters_data:
            return set()

        newsletters = set(
            Newsletter(
                self.client,
                id=n.get("id"),
                name=n.get("name"),
                slug=n.get("slug"),
                description=n.get("description"),
                status=n.get("status"),
            )
            for n in newsletters_data
        )

        for newsletter in newsletters:
            newsletter.exists = True

        return newsletters

    def _get_labels(self, data: dict):
        labels_data = data.get("labels")
        if not labels_data:
            return set()

        labels = set(
            Label(
                self.client,
                id=label.get("id"),
                slug=label.get("slug"),
                name=label.get("name"),
            )
            for label in labels_data
        )

        for label in labels:
            label.exists = True

        return labels

    def _get_tiers(self, data: dict):
        tiers_data = data.get("tiers")
        if not tiers_data or len(tiers_data) == 0:
            subscriptions = data.get("subscriptions")
            if not subscriptions or len(subscriptions) == 0:
                return set()
            tiers_data = [
                s["tier"]
                for s in subscriptions
                if s["status"] in ("active", "past_due")
            ]

        if not tiers_data or len(tiers_data) == 0:
            return set()

        tiers = set(
            Tier(
                self.client,
                id=t.get("id"),
                slug=t.get("slug"),
                name=t.get("name"),
                active=t.get("active"),
                welcome_page_url=t.get("welcome_page_url"),
                visibility=t.get("visibility"),
                trial_days=t.get("trial_days"),
                description=t.get("description"),
                tier_type=(TierType(t["type"]) if t.get("type") else TierType.FREE),
                currency=t.get("currency"),
                monthly_price=t.get("monthly_price"),
                yearly_price=t.get("yearly_price"),
                benefits=t.get("benefits"),
                created_at=datetime.fromisoformat(t.get("created_at", "")),
                updated_at=datetime.fromisoformat(t.get("updated_at", "")),
            )
            for t in tiers_data
        )
        for tier in tiers:
            tier.exists = True

        return tiers
