from ghost.client import AdminAPIClient
from ghost.exceptions import BadRequestException, GhostException, NotFoundException
from ghost.models import BaseModel
from datetime import datetime
from typing import Optional


class Newsletter(BaseModel):
    printable: list[str] = ["id", "name", "slug"]

    client: AdminAPIClient
    id: str | None
    name: str | None

    def __init__(
        self,
        client: AdminAPIClient,
        id: Optional[str] = None,
        name: Optional[str] = None,
        slug: Optional[str] = None,
        description: Optional[str] = None,
        sender_name: Optional[str] = None,
        sender_email: Optional[str] = None,
        sender_reply_to: Optional[str] = None,
        status: str = "archived",
        visibility: str = "members",
    ):
        super().__init__()

        if not id and not slug:
            raise GhostException("One of `id` or `slug` must be passed")

        self.client: AdminAPIClient = client
        self.id: str | None = id
        self.name: str | None = name
        self.slug: str | None = slug
        self.description: str | None = description
        self.sender_name: str | None = sender_name
        self.sender_email: str | None = sender_email
        self.sender_reply_to: str | None = sender_reply_to
        self.status: str = status
        self.visibility: str = visibility

    def get(self):
        try:
            response = self.client.get(
                "/newsletters/", params={"filter": f"id:{self.id}"}
            )
            if (
                not response
                or not response["newsletters"]
                or not len(response["newsletters"])
            ):
                raise BadRequestException("/newsletters/", 404, "")
        except BadRequestException as e:
            if e.status_code != 404:
                raise
            response = self.client.get(
                "/newsletters/", params={"filter": f"slug:{self.slug}"}
            )

        if (
            not response
            or not response["newsletters"]
            or not len(response["newsletters"])
        ):
            raise NotFoundException(self, {"id": self.id, "slug": self.slug})

        self._assign_from_data(response["newsletters"][0])
        super().get()
        return self

    def create(self):
        super().create()

        newsletter_data = self.to_dict()
        del newsletter_data["id"]  # ID should be read-only

        response = self.client.post(
            "/newsletters/", payload={"newsletters": [newsletter_data]}
        )
        self._assign_from_data(response["newsletters"][0])
        self.exists = True

        return self

    def update(self):
        super().update()

        newsletter_data = self.to_dict()
        del newsletter_data["id"]  # ID should be read-only

        response = self.client.put(
            f"/newsletters/{self.id}", payload={"newsletters": [newsletter_data]}
        )

        if (
            not response
            or not response["newsletters"]
            or not len(response["newsletters"])
        ):
            raise GhostException(
                "Could not update newsletter: empty answer from server."
            )

        self._assign_from_data(response["newsletters"][0])

        return self

    def to_dict(self) -> dict:
        # Not everything can be created or updated for now
        dict_version = {
            "id": self.id,
            "slug": self.slug,
            "name": self.name,
            "description": self.description,
            "status": self.status,
            "visibility": self.visibility,
        }

        return dict_version

    def summarize(self) -> dict:
        return {"id": self.id, "slug": self.slug}

    def _assign_from_data(self, data: dict):
        self.id = data.get("id")
        self.name = data.get("name")
        self.slug = data.get("slug")
        self.description = data.get("description")
        self.sender_name = data.get("sender_name")
        self.sender_email = data.get("sender_email")
        self.sender_reply_to = data.get("sender_reply_to")
        self.status = data.get("status", "archived")
        self.visibility = data.get("visibility", "members")
        self.subscribe_on_signup = data.get("subscribe_on_signup", False)
        self.show_header_icon = data.get("show_header_icon", False)
        self.show_header_type = data.get("show_header_type", False)
        self.title_font_category = data.get("title_font_category")
        self.title_alignment = data.get("title_alignment")
        self.show_feature_image = data.get("show_feature_image", False)
        self.body_font_category = data.get("body_font_category")
        self.footer_content = data.get("footer_content")
        self.show_badge = data.get("show_badge", False)
        self.show_header_name = data.get("show_header_name", False)
        self.created_at = datetime.fromisoformat(data.get("created_at", ""))
        self.updated_at = datetime.fromisoformat(data.get("updated_at", ""))
        self.feedback_enabled = data.get("feedback_enabled", False)
        self.show_post_title_section = data.get("show_post_title_section", False)
        self.show_comment_cta = data.get("show_comment_cta", False)
        self.show_subscription_details = data.get("show_subscription_details", False)
        self.show_latest_posts = data.get("show_latest_posts")
        self.background_color = data.get("background_color")
        self.border_color = data.get("border_color")
        self.title_color = data.get("title_color")
        self.show_excerpt = data.get("show_excerpt", False)

    def __hash__(self) -> int:
        return hash((self.id, self.slug))


if __name__ == "__main__":
    from dotenv import load_dotenv
    from os import getenv

    load_dotenv()

    client = AdminAPIClient(getenv("GHOST_API_TOKEN", ""), getenv("GHOST_URL", ""))
    newsletter = Newsletter(client, "66cce175a306c0006676faf2").get()
    print(newsletter.to_dict())
