from ghost.models.base_model import BaseModel
from ghost.client import APIClient, AdminAPIClient, ContentAPIClient
from ghost.exceptions import GhostException, BadRequestException, NotFoundException
from typing import Optional
from datetime import datetime


class Post(BaseModel):
    printable = []

    def __init__(
        self, client: APIClient, id: Optional[str] = None, slug: Optional[str] = None
    ):
        super().__init__()
        self.client = client
        self.id = id
        self.slug = slug

    def get(self):
        if self.id is None and self.slug is None:
            raise GhostException("Cannot retrieve member with no email or id.")

        response = {}

        if self.id:
            try:
                response = self.client.get(f"/posts/{self.id}")
            except BadRequestException as e:
                if e.status_code == 404:
                    response = {}
                else:
                    raise

        if self.slug and not response:
            try:
                response = self.client.get(f"/posts/slug/{self.slug}")
            except BadRequestException as e:
                if e.status_code == 404:
                    raise NotFoundException(self, {"id": self.id, "slug": self.slug})
                else:
                    raise
        elif not response:
            raise GhostException(
                f"No post found with given ID ({self.id}) and slug ({self.slug})"
            )

        self._assign_from_data(response["posts"][0])

        super().get()
        return self

    def create(self):
        super().create()

        if self.title is None:
            raise GhostException("Cannot create post without title")

        post_data = {"title": self.title}
        if self.slug:
            post_data["slug"] = self.slug
        if self.lexical:
            post_data["lexical"] = self.lexical
        if not self.lexical and self.html:
            post_data["html"] = self.html
        if self.feature_image:
            post_data["feature_image"] = self.feature_image
        if self.feature_image_caption:
            post_data["feature_image_caption"] = self.feature_image_caption
        if self.feature_image_alt:
            post_data["feature_image_alt"] = self.feature_image_alt
        if self.visibility:
            post_data["visibility"] = self.visibility
        if self.excerpt:
            post_data["excerpt"] = self.excerpt
        if self.email_subject:
            post_data["email_subject"] = self.email_subject

        params = {}
        if post_data.get("html"):
            params["source"] = "html"

        response = self.client.post("/posts/", payload=post_data, params=params)
        self._assign_from_data(response["members"][0])
        self.exists = True

        return self

    def copy(self):
        if not self.exists:
            raise GhostException("Cannot copy an unexisting post")

        if not self.id:
            raise GhostException("A valid ID is required for copying a post")

        new_post = Post(self.client)

        response = self.client.post(f"/posts/{self.id}/copy")
        new_post._assign_from_data(response)
        new_post.exists = True

        return new_post

    def update(self):
        super().update()

        post_data = {"updated_at": datetime.now().isoformat()}
        if self.slug:
            post_data["slug"] = self.slug
        if self.lexical:
            post_data["lexical"] = self.lexical
        if not self.lexical and self.html:
            post_data["html"] = self.html
        if self.feature_image:
            post_data["feature_image"] = self.feature_image
        if self.feature_image_caption:
            post_data["feature_image_caption"] = self.feature_image_caption
        if self.feature_image_alt:
            post_data["feature_image_alt"] = self.feature_image_alt
        if self.visibility:
            post_data["visibility"] = self.visibility
        if self.excerpt:
            post_data["excerpt"] = self.excerpt
        if self.email_subject:
            post_data["email_subject"] = self.email_subject

        params = {}
        if post_data.get("html"):
            params["source"] = "html"

        response = self.client.post("/posts/", payload=post_data, params=params)
        self._assign_from_data(response["members"][0])
        self.exists = True

        return self

    def _assign_from_data(self, data: dict):
        self.id = data.get("id")
        self.slug = data.get("slug")
        self.uuid = data.get("uuid")
        self.title = data.get("title")
        self.html = data.get("html")
        self.comment_id = data.get("comment_id")
        self.feature_image = data.get("feature_image")
        self.feature_image_alt = data.get("feature_image_alt")
        self.feature_image_caption = data.get("feature_image_caption")
        self.featured = data.get("featured")
        self.visibility = data.get("visibility")
        self.created_at = datetime.fromisoformat(str(data.get("created_at")))
        self.updated_at = datetime.fromisoformat(str(data.get("updated_at")))
        self.published_at = (
            datetime.fromisoformat(str(data.get("published_at")))
            if data.get("published_at")
            else None
        )
        self.url = data.get("url")
        self.excerpt = data.get("excerpt")
        self.reading_time = data.get("reading_time")
        self.email_subject = data.get("email_subject")
