from enum import Enum
from ghost.client import AdminAPIClient
from ghost.exceptions import BadRequestException, GhostException, NotFoundException
from ghost.models import BaseModel
from datetime import datetime
from typing import Optional


class TierType(str, Enum):
    FREE = "free"
    PAID = "paid"


class Tier(BaseModel):
    printable = [
        "id",
        "name",
        "slug",
        "active",
        "welcome_page_url",
        "visibility",
        "tier_type",
    ]

    def __init__(
        self,
        client: AdminAPIClient,
        id: Optional[str] = None,
        name: Optional[str] = None,
        slug: Optional[str] = None,
        active: bool = False,
        welcome_page_url: Optional[str] = None,
        visibility: Optional[str] = None,
        trial_days: Optional[int] = None,
        description: Optional[str] = None,
        tier_type: TierType = TierType.FREE,
        currency: Optional[str] = None,
        monthly_price: int = 0,
        yearly_price: int = 0,
        benefits: Optional[list[str]] = None,
        created_at: Optional[datetime] = None,
        updated_at: Optional[datetime] = None,
    ):
        super().__init__()

        if not id and not slug:
            raise GhostException("One of `id` or `slug` must be passed")

        self.client: AdminAPIClient = client
        self.id: str | None = id
        self.name: str | None = name
        self.slug: str | None = slug
        self.active: bool = active
        self.welcome_page_url: str | None = welcome_page_url
        self.visibility: str | None = visibility
        self.trial_days: int | None = trial_days
        self.description: str | None = description
        self.tier_type: TierType = tier_type
        self.currency: str | None = currency
        self.monthly_price: int = monthly_price
        self.yearly_price: int = yearly_price
        self.created_at: datetime | None = created_at
        self.updated_at: datetime | None = updated_at
        self.benefits: list[str] = []
        if benefits is not None:
            self.benefits = benefits

    def get(self):
        try:
            response = self.client.get(
                "/tiers/",
                params={
                    "filter": f"id:{self.id}",
                    "include": "monthly_price,yearly_price,benefits",
                },
            )
            if not response or not response["tiers"] or not len(response["tiers"]):
                raise BadRequestException("/tiers/", 404, "")
        except BadRequestException as e:
            if e.status_code != 404:
                raise
            response = self.client.get(
                "/tiers/",
                params={
                    "filter": f"slug:{self.slug}",
                    "include": "monthly_price,yearly_price,benefits",
                },
            )

        if not response or not response["tiers"] or not len(response["tiers"]):
            raise NotFoundException(self, {"id": self.id, "slug": self.slug})

        self._assign_from_data(response["tiers"][0])
        super().get()
        return self

    def create(self):
        super().create()

        tier_data = self.to_dict()
        del tier_data["id"]  # ID should be read-only

        response = self.client.post("/tiers/", payload={"tiers": [tier_data]})
        self._assign_from_data(response["tiers"][0])
        self.exists = True

        return self

    def update(self):
        super().update()

        tier_data = self.to_dict()
        del tier_data["id"]  # ID should be read-only

        response = self.client.put(f"/tiers/{self.id}", payload={"tiers": [tier_data]})

        if not response or not response["tiers"] or not len(response["tiers"]):
            raise GhostException("Could not update tier: empty answer from server.")

        self._assign_from_data(response["tiers"][0])

        return self

    def to_dict(self) -> dict:
        dict_version = {
            "id": self.id,
            "slug": self.slug,
            "name": self.name,
            "active": self.active,
            "welcome_page_url": self.welcome_page_url,
            "visibility": self.visibility,
            "trial_days": self.trial_days,
            "description": self.description,
            "type": self.tier_type.value,
            "currency": self.currency,
            "monthly_price": self.monthly_price,
            "yearly_price": self.yearly_price,
            "benefits": self.benefits,
        }

        return dict_version

    def summarize(self) -> dict:
        return {"id": self.id, "slug": self.slug}

    def _assign_from_data(self, data: dict):
        self.id = data.get("id")
        self.name = data.get("name")
        self.slug = data.get("slug")
        self.active = data.get("active", False)
        self.welcome_page_url = data.get("welcome_page_url")
        self.visibility = data.get("visibility")
        self.trial_days = data.get("trial_days", 0)
        self.description = data.get("description")
        self.tier_type = TierType(data.get("type", "free"))
        self.currency = data.get("currency")
        self.monthly_price = data.get("monthly_price", 0)
        self.yearly_price = data.get("yearly_price", 0)
        self.benefits = data.get("benefits", [])
        self.created_at = datetime.fromisoformat(str(data.get("created_at")))
        self.updated_at = datetime.fromisoformat(str(data.get("updated_at")))

    # In Ghost, Tiers are naturally ordered by monthly price.
    # The following functions reflect this behaviour so that
    # Tiers can be sorted in the code as well.
    def __eq__(self, other, /) -> bool:
        return self.monthly_price == other.monthly_price

    def __gt__(self, other, /) -> bool:
        return self.monthly_price > other.monthly_price

    def __ge__(self, other, /) -> bool:
        return self.monthly_price >= other.monthly_price

    def __lt__(self, other, /) -> bool:
        return self.monthly_price < other.monthly_price

    def __le__(self, other, /) -> bool:
        return self.monthly_price <= other.monthly_price

    def __ne__(self, other, /) -> bool:
        return not self.__eq__(other)

    def __hash__(self) -> int:
        return hash((self.id, self.slug))


if __name__ == "__main__":
    from dotenv import load_dotenv
    from os import getenv

    load_dotenv()

    client = AdminAPIClient(getenv("GHOST_API_TOKEN", ""), getenv("GHOST_URL", ""))
    tier = Tier(client, "66b64c6ab0da39005e56f114").get()
    print(tier.to_dict())
