from fastapi import Header, Request, HTTPException
from pydantic import BaseModel
from os import getenv
import hmac
import json
import re


class MemberData(BaseModel):
    member: dict


class WebhookAuth:
    def __init__(self, secret_key: str = "GHOST_WEBHOOK_SECRET"):
        self.secret_key = secret_key

    async def __call__(
        self,
        request: Request,
        x_ghost_signature: str = Header(),
    ):
        if not x_ghost_signature:
            raise HTTPException(status_code=403)

        hmac_header_search = re.search("sha256=(.*), t=([0-9]*)", x_ghost_signature)
        if not hmac_header_search:
            raise HTTPException(status_code=403)

        sha256, timestamp = hmac_header_search.group(1, 2)

        raw_input = await request.body()
        input_hmac = hmac.new(
            key=getenv(self.secret_key, "").encode(),
            msg=raw_input + timestamp.encode(),
            digestmod="sha256",
        )

        if not hmac.compare_digest(input_hmac.hexdigest(), sha256):
            raise HTTPException(
                status_code=401,
                detail=f"expected {input_hmac.hexdigest()}, got {sha256}",
            )

        return json.loads(raw_input)
