from unittest import TestCase, main
from ghost import AdminAPIClient
from ghost.exceptions import BadRequestException, GhostException
from ghost.models import Label
from faker import Faker
from datetime import datetime
from dotenv import load_dotenv
from os import getenv

load_dotenv()


class TestLabel(TestCase):
    fake = Faker()
    client = AdminAPIClient(getenv("GHOST_API_TOKEN", ""), getenv("GHOST_URL", ""))

    def test_init(self):
        with self.assertRaises(GhostException):
            Label(self.client)

        id = "1235"
        slug = "dfdfbdf"

        label = Label(self.client, id=id)
        self.assertFalse(label.exists)
        self.assertEqual(label.id, id)
        self.assertIsNone(label.slug, slug)

        label = Label(self.client, slug=slug)
        self.assertFalse(label.exists)
        self.assertIsNone(label.id, id)
        self.assertEqual(label.slug, slug)

        label = Label(self.client, id=id, slug=slug)
        self.assertFalse(label.exists)
        self.assertEqual(label.id, id)
        self.assertEqual(label.slug, slug)

    def test_get(self):
        id = "63e9160fca774efb785b279f"
        slug = "test"

        # Try with invalid ID
        label = Label(self.client, id="0000000")
        with self.assertRaises(GhostException):
            label.get()
        self.assertFalse(label.exists)

        # Retry with valid slug
        label.slug = slug
        label.get()

        # Try getting label by ID and slug
        label_by_id = Label(self.client, id=id).get()
        label_by_slug = Label(self.client, slug=slug).get()

        # All labels should exist on Ghost
        self.assertTrue(label.exists)
        self.assertTrue(label_by_id.exists)
        self.assertTrue(label_by_slug.exists)

        # All IDs should be equal
        self.assertEqual(label.id, label_by_id.id)
        self.assertEqual(label.id, label_by_slug.id)

        # All slugs should be equal
        self.assertEqual(label.slug, label_by_id.slug)
        self.assertEqual(label.slug, label_by_slug.slug)

        # ID and slug should be the provided ones
        self.assertEqual(label.id, id)
        self.assertEqual(label.slug, slug)

        # Other information should be fetched from API as well
        self.assertEqual(label.name, "test")
        self.assertTrue(isinstance(label.created_at, datetime))
        self.assertTrue(isinstance(label.updated_at, datetime))

    def test_create(self):
        slug = name = self.fake.word()
        label = Label(self.client, slug=slug, name=name)

        # Cannot get label before creation
        with self.assertRaises(GhostException):
            label.get()

        # Cannot create without slug or name
        with self.assertRaises(GhostException):
            Label(self.client).create()

        # Now try creating with valid slug and name
        label.create()
        self.assertTrue(label.exists)
        self.assertIsNotNone(label.id)
        self.assertEqual(label.name, name)
        self.assertEqual(label.slug, slug)

        # Creating should allow fetching
        label.get()

        # Cannot create label again
        with self.assertRaises(GhostException):
            label.create()

        # Even if forced to
        label.exists = False
        with self.assertRaises(BadRequestException):
            label.create()
        label.exists = True

    def test_update(self):
        slug = self.fake.word()
        label = Label(self.client, slug=slug)

        # Cannot update a label that doesn't exist
        with self.assertRaises(GhostException):
            label.update()

        # Even if forced to
        label.exists = True
        with self.assertRaises(BadRequestException):
            label.update()
        label.exists = False

        # Get valid label
        slug = "test"
        label = Label(self.client, slug=slug).get()

        # Retrieve info
        id = label.id

        # Update name
        old_name = label.name
        label.name = self.fake.word()
        label.update()

        # Check if everything is ok
        self.assertEqual(label.id, id)
        self.assertEqual(label.slug, slug)
        self.assertNotEqual(label.name, old_name)

        # Now try updating slug
        label.slug = self.fake.word()
        label.update()
        self.assertEqual(label.id, id)
        self.assertNotEqual(label.slug, slug)
        self.assertNotEqual(label.name, old_name)

        # Revert label data
        label.slug = slug
        label.name = old_name
        label.update()

    def test_dict(self):
        id = "63e9160fca774efb785b279f"
        slug = "test"
        label = Label(self.client, id=id, slug=slug)

        label_dict = label.to_dict()
        self.assertEqual(set(label_dict.keys()), {"id", "name", "slug"})
        self.assertEqual(label_dict["id"], id)
        self.assertIsNone(label_dict["name"])
        self.assertEqual(label_dict["slug"], slug)

    def test_summarize(self):
        id = "63e9160fca774efb785b279f"
        slug = "test"
        label = Label(self.client, id=id, slug=slug)

        label_summary = label.summarize()
        self.assertEqual(set(label_summary.keys()), {"id", "slug"})
        self.assertEqual(label_summary["id"], id)
        self.assertEqual(label_summary["slug"], slug)


if __name__ == "__main__":
    main()
