from unittest import TestCase, main
from ghost import AdminAPIClient, Member
from ghost.exceptions import GhostException, BadRequestException
from ghost.filter import Query, Operator
from faker import Faker
from dotenv import load_dotenv
from os import getenv
from datetime import datetime, timedelta


load_dotenv()


class TestMember(TestCase):
    fake = Faker()
    client = AdminAPIClient(getenv("GHOST_API_TOKEN", ""), getenv("GHOST_URL", ""))

    def test_init(self):
        email = self.fake.email()
        member = Member(self.client, email=email)
        self.assertFalse(member.exists)
        self.assertEqual(member.email, email)
        self.assertIsNone(member.id)

    def test_get(self):
        id = "641c7e90387ed3069bf399af"
        email = "teste@voltdata.info"

        # Try with invalid ID
        member_error = Member(self.client, id="0000000")
        with self.assertRaises(GhostException):
            member_error.get()
        self.assertFalse(member_error.exists)

        # Retry with valid email
        member_error.email = email
        member_error.get()

        # Try getting member by ID and email
        member_id = Member(self.client, id=id).get()
        member_email = Member(self.client, email=email).get()

        # All subscribers should exist on Ghost
        self.assertTrue(member_error.exists)
        self.assertTrue(member_id.exists)
        self.assertTrue(member_email.exists)

        # All IDs should be equal
        self.assertEqual(member_id.id, member_email.id)
        self.assertEqual(member_id.id, member_error.id)

        # All emails should be equal
        self.assertEqual(member_email.email, member_id.email)
        self.assertEqual(member_email.email, member_error.email)

        # ID and email should be the provided ones
        self.assertEqual(member_id.id, id)
        self.assertEqual(member_email.email, email)

        # Other information should be pulled from the API as well
        self.assertEqual(len(member_id.newsletters), 5)
        self.assertEqual(len(member_id.labels), 2)
        self.assertEqual(
            member_id.avatar_image,
            "https://www.gravatar.com/avatar/510d3ae77c843c1621007369fb01858c?s=250&r=g&d=blank",
        )
        self.assertEqual(
            member_id.created_at, datetime.fromisoformat("2023-03-23T16:30:08.000Z")
        )

    def test_create(self):
        email = self.fake.email()
        member = Member(self.client, email=email, note="Test member")

        # Cannot retrieve member before creation
        with self.assertRaises(GhostException):
            member.get()

        member.create()

        # Member should now exist
        self.assertTrue(member.exists)
        self.assertTrue(member.id)

        # Member email should be the provided one
        self.assertEqual(member.email, email)

        # After creating, should be able to retrieve information
        member.get()

        # Cannot create member again
        with self.assertRaises(GhostException):
            member.create()

        # Even if forced to
        member.exists = False
        with self.assertRaises(BadRequestException):
            member.create()
        member.exists = True

        # API does not have support for deletion just now :(

    def test_update(self):
        # Cannot update a member that doesn't exist
        member = Member(self.client, email=self.fake.email())
        with self.assertRaises(GhostException):
            member.update()

        # Even if forced to
        member.exists = True
        with self.assertRaises(BadRequestException):
            member.update()
        member.exists = False

        # Get a valid member then
        email = "teste@voltdata.info"
        member = Member(self.client, email=email).get()

        # Retrieve info from it
        id = member.id

        # Update some
        update_date = datetime.now()
        message = f"\nTeste update {update_date.isoformat()}"

        member.note += message
        member.update()

        # Check everthing is ok
        self.assertTrue(member.exists)
        self.assertEqual(member.id, id)
        self.assertEqual(member.email, email)
        self.assertLessEqual(
            abs(member.updated_at - datetime.now(tz=member.updated_at.tzinfo)),
            timedelta(seconds=2),
        )
        self.assertTrue(member.note.endswith(message))

        # Try with email as well
        # Apparently not possible
        # new_email = self.fake.email()
        # member.email = new_email
        # member.update()
        # self.assertEqual(member.email, new_email)

        # Reset to old email
        member.email = email
        member.update()
        self.assertEqual(member.email, email)

    def test_add_remove_newsletters(self):
        member = Member(self.client, email="teste@voltdata.info").get()

        # Copy the member newsletters for future testing
        old_newsletters = member.newsletters.copy()

        # Try adding something that isn't valid
        with self.assertRaises(GhostException):
            member.add_newsletters(dict())

        # Try removing something that isn't valid
        with self.assertRaises(GhostException):
            member.remove_newsletters(dict())

        # Add a newsletter by string
        member.add_newsletters("6679c963ddf78e005e5e735e")
        self.assertTrue(
            "6679c963ddf78e005e5e735e" in [n.id for n in member.newsletters]
        )

        # Now remove the newsletter
        member.remove_newsletters("6679c963ddf78e005e5e735e")
        self.assertTrue(
            "6679c963ddf78e005e5e735e" not in [n.id for n in member.newsletters]
        )

        # Add newsletters by list
        member.add_newsletters(["6679c963ddf78e005e5e735e", "66619a58146113005fcb4ee8"])
        self.assertTrue(
            "6679c963ddf78e005e5e735e" in [n.id for n in member.newsletters]
        )
        self.assertTrue(
            "66619a58146113005fcb4ee8" in [n.id for n in member.newsletters]
        )

        # Now remove them by list
        member.remove_newsletters(
            ["6679c963ddf78e005e5e735e", "66619a58146113005fcb4ee8"]
        )
        self.assertTrue(
            "6679c963ddf78e005e5e735e" not in [n.id for n in member.newsletters]
        )
        self.assertTrue(
            "66619a58146113005fcb4ee8" not in [n.id for n in member.newsletters]
        )

        # Add by set
        member.add_newsletters(
            set(["6679c963ddf78e005e5e735e", "66619a58146113005fcb4ee8"])
        )
        self.assertTrue(
            "6679c963ddf78e005e5e735e" in [n.id for n in member.newsletters]
        )
        self.assertTrue(
            "66619a58146113005fcb4ee8" in [n.id for n in member.newsletters]
        )

        # Remove by set
        member.remove_newsletters(
            set(["6679c963ddf78e005e5e735e", "66619a58146113005fcb4ee8"])
        )
        self.assertTrue(
            "6679c963ddf78e005e5e735e" not in [n.id for n in member.newsletters]
        )
        self.assertTrue(
            "66619a58146113005fcb4ee8" not in [n.id for n in member.newsletters]
        )

        # Finally, newsletters should be the same as when this started
        self.assertSetEqual(member.newsletters, old_newsletters)

    def test_add_remove_tiers(self):
        member = Member(self.client, email="teste@voltdata.info").get()

        # Copy the member tiers for future testing
        old_tiers = member.tiers.copy()

        # Try adding something that isn't valid
        with self.assertRaises(GhostException):
            member.add_tiers(dict())

        # Try removing something that isn't valid
        with self.assertRaises(GhostException):
            member.remove_tiers(dict())

        # Add a tier by string
        member.add_tiers("6294176b84b84878740ac7f8")
        self.assertTrue("6294176b84b84878740ac7f8" in [t.id for t in member.tiers])

        # Now remove the tier
        member.remove_tiers("6294176b84b84878740ac7f8")
        self.assertTrue("6294176b84b84878740ac7f8" not in [t.id for t in member.tiers])

        # Add tiers by list
        member.add_tiers(["6294176b84b84878740ac7f8"])
        self.assertTrue("6294176b84b84878740ac7f8" in [t.id for t in member.tiers])

        # Now remove them by list
        member.remove_tiers(["6294176b84b84878740ac7f8"])
        self.assertTrue("6294176b84b84878740ac7f8" not in [t.id for t in member.tiers])

        # Add by set
        member.add_tiers(set(["6294176b84b84878740ac7f8"]))
        self.assertTrue("6294176b84b84878740ac7f8" in [t.id for t in member.tiers])

        # Remove by set
        member.remove_tiers(set(["6294176b84b84878740ac7f8"]))
        self.assertTrue("6294176b84b84878740ac7f8" not in [t.id for t in member.tiers])

        # Finally, newsletters should be the same as when this started
        self.assertSetEqual(member.tiers, old_tiers)

    def test_send_magic_link(self):
        member = Member(self.client, email="henrique@voltdata.info")

        # Cannot send magic link to unsynced member
        with self.assertRaises(GhostException):
            member.send_magic_link()

        # Should send to existing synchronized member with valid email
        member.get().send_magic_link()

    def test_all(self):
        members = Member.all(self.client)
        self.assertGreater(len(members), 0)

        members = Member.all(self.client, pagesize=5, limit=10)
        self.assertEqual(len(members), 10)

        members = Member.all(self.client, filters={"email": "henrique@voltdata.info"})
        self.assertEqual(len(members), 1)
        self.assertEqual(members[0].email, "henrique@voltdata.info")

        members = Member.all(
            self.client, filters=Query("email", "@voltdata.info", Operator.ENDS_WITH)
        )
        self.assertGreater(len(members), 0)
        for member in members:
            self.assertRegex(member.email, "@voltdata.info$")

        members = Member.all(
            self.client,
            filters=Query("email", ["henrique@voltdata.info", "sergio@voltdata.info"]),
        )
        self.assertEqual(len(members), 2)
        for member in members:
            self.assertIn(
                member.email, ["henrique@voltdata.info", "sergio@voltdata.info"]
            )


if __name__ == "__main__":
    main()
