from unittest import TestCase, main
from ghost import AdminAPIClient
from ghost.exceptions import BadRequestException, GhostException
from ghost.models import Newsletter
from faker import Faker
from datetime import datetime
from dotenv import load_dotenv
from os import getenv

load_dotenv()


class TestNewsletter(TestCase):
    fake = Faker()
    client = AdminAPIClient(getenv("GHOST_API_TOKEN", ""), getenv("GHOST_URL", ""))

    def test_init(self):
        with self.assertRaises(GhostException):
            Newsletter(self.client)

        id = "1235"
        slug = "dfdfbdf"

        newsletter = Newsletter(self.client, id=id)
        self.assertFalse(newsletter.exists)
        self.assertEqual(newsletter.client, self.client)
        self.assertEqual(newsletter.id, id)
        self.assertIsNone(newsletter.slug, slug)

        newsletter = Newsletter(self.client, slug=slug)
        self.assertFalse(newsletter.exists)
        self.assertEqual(newsletter.client, self.client)
        self.assertIsNone(newsletter.id, id)
        self.assertEqual(newsletter.slug, slug)

        newsletter = Newsletter(self.client, id=id, slug=slug)
        self.assertFalse(newsletter.exists)
        self.assertEqual(newsletter.client, self.client)
        self.assertEqual(newsletter.id, id)
        self.assertEqual(newsletter.slug, slug)

    def test_get(self):
        id = "66cce175a306c0006676faf2"
        slug = "anuncios-de-membresia"

        # Try with invalid ID
        newsletter = Newsletter(self.client, id="0000000")
        with self.assertRaises(GhostException):
            newsletter.get()
        self.assertFalse(newsletter.exists)

        # Retry with valid slug
        newsletter.slug = slug
        newsletter.get()

        # Try getting newsletter by ID and slug
        newsletter_by_id = Newsletter(self.client, id=id).get()
        newsletter_by_slug = Newsletter(self.client, slug=slug).get()

        # All newsletters should exist on Ghost
        self.assertTrue(newsletter.exists)
        self.assertTrue(newsletter_by_id.exists)
        self.assertTrue(newsletter_by_slug.exists)

        # All IDs should be equal
        self.assertEqual(newsletter.id, newsletter_by_id.id)
        self.assertEqual(newsletter.id, newsletter_by_slug.id)

        # All slugs should be equal
        self.assertEqual(newsletter.slug, newsletter_by_id.slug)
        self.assertEqual(newsletter.slug, newsletter_by_slug.slug)

        # ID and slug should be the provided ones
        self.assertEqual(newsletter.id, id)
        self.assertEqual(newsletter.slug, slug)

        # Other information should be fetched from API as well
        self.assertEqual(newsletter.name, "Anúncios para comunidade")
        self.assertEqual(len(str(newsletter.description)), 88)
        self.assertIsNone(newsletter.sender_name)
        self.assertEqual(newsletter.sender_email, "comunidade@nucleo.jor.br")
        self.assertEqual(newsletter.sender_reply_to, "comunidade@nucleo.jor.br")
        self.assertEqual(newsletter.status, "active")
        self.assertEqual(newsletter.visibility, "members")
        self.assertTrue(newsletter.subscribe_on_signup)
        self.assertFalse(newsletter.show_header_icon)
        self.assertTrue(isinstance(newsletter.created_at, datetime))
        self.assertTrue(isinstance(newsletter.updated_at, datetime))

    def test_create(self):
        slug = name = self.fake.word()
        description = " ".join(self.fake.words(10))
        newsletter = Newsletter(
            self.client,
            slug=slug,
            name=name,
            description=description,
            status="archived",
        )

        # Cannot get newsletter before creation
        with self.assertRaises(GhostException):
            newsletter.get()

        # Cannot create without slug or name
        with self.assertRaises(GhostException):
            Newsletter(self.client).create()

        # Now try creating with valid slug and name
        newsletter.create()
        self.assertTrue(newsletter.exists)
        self.assertIsNotNone(newsletter.id)
        self.assertEqual(newsletter.name, name)
        self.assertEqual(newsletter.slug, slug)

        # Creating should allow fetching
        newsletter.get()

        # Cannot create newsletter again
        with self.assertRaises(GhostException):
            newsletter.create()

        # Even if forced to
        newsletter.exists = False
        with self.assertRaises(BadRequestException):
            newsletter.create()
        newsletter.exists = True

    def test_update(self):
        slug = self.fake.word()
        newsletter = Newsletter(self.client, slug=slug)

        # Cannot update a newsletter that doesn't exist
        with self.assertRaises(GhostException):
            newsletter.update()

        # Even if forced to
        newsletter.exists = True
        with self.assertRaises(BadRequestException):
            newsletter.update()
        newsletter.exists = False

        # Get valid newsletter
        slug = "anuncios-de-membresia"
        newsletter = Newsletter(self.client, slug=slug).get()

        # Retrieve info
        id = newsletter.id

        # Update name
        old_name = newsletter.name
        newsletter.name = self.fake.word()
        newsletter.update()

        # Check if everything is ok
        self.assertEqual(newsletter.id, id)
        self.assertEqual(newsletter.slug, slug)
        self.assertNotEqual(newsletter.name, old_name)

        # Now try updating slug
        newsletter.slug = self.fake.word()
        newsletter.update()
        self.assertEqual(newsletter.id, id)
        self.assertNotEqual(newsletter.slug, slug)
        self.assertNotEqual(newsletter.name, old_name)

        # Revert newsletter data
        newsletter.slug = slug
        newsletter.name = old_name
        newsletter.update()

    def test_dict(self):
        id = "66cce175a306c0006676faf2"
        slug = "anuncios-de-membresia"
        name = "Anúncios para comunidade"
        newsletter = Newsletter(self.client, id=id, slug=slug).get()

        newsletter_dict = newsletter.to_dict()
        self.assertEqual(
            set(newsletter_dict.keys()),
            {
                "id",
                "name",
                "slug",
                "description",
                "status",
                "visibility",
            },
        )
        self.assertEqual(newsletter_dict["id"], id)
        self.assertEqual(newsletter_dict["name"], name)
        self.assertEqual(newsletter_dict["slug"], slug)

    def test_summarize(self):
        id = "63e9160fca774efb785b279f"
        slug = "test"
        newsletter = Newsletter(self.client, id=id, slug=slug)

        newsletter_summary = newsletter.summarize()
        self.assertEqual(set(newsletter_summary.keys()), {"id", "slug"})
        self.assertEqual(newsletter_summary["id"], id)
        self.assertEqual(newsletter_summary["slug"], slug)


if __name__ == "__main__":
    main()
