from unittest import TestCase, main
from ghost import ContentAPIClient, AdminAPIClient, Post
from ghost.exceptions import GhostException, BadRequestException
from faker import Faker
from dotenv import load_dotenv
from os import getenv


load_dotenv()


class TestPost(TestCase):
    def __init__(self, arg):
        super().__init__(arg)

        self.fake = Faker()
        self.contentClient = ContentAPIClient(
            getenv("GHOST_CONTENT_TOKEN", ""), getenv("GHOST_URL", "")
        )
        self.adminClient = AdminAPIClient(
            getenv("GHOST_API_TOKEN", ""), getenv("GHOST_URL", "")
        )

    def test_init(self):
        slug = self.fake.text()
        post = Post(self.contentClient, slug=slug)
        self.assertEqual(post.exists, False)
        self.assertEqual(post.client, self.contentClient)
        self.assertEqual(post.slug, slug)
        self.assertIsNone(post.id)

    def test_get(self):
        # Try without passing ID or slug
        with self.assertRaises(GhostException):
            Post(self.adminClient).get()

        # Try with invalid ID
        with self.assertRaises(BadRequestException):
            Post(self.adminClient, id="1234").get()  # Fails ID validation

        with self.assertRaises(GhostException):
            Post(
                self.contentClient, id="677fbd106592cb0065315619"
            ).get()  # ID doesn't exist

        # Try with invalid slug
        with self.assertRaises(GhostException):
            Post(self.adminClient, slug="batatinhafrita123").get()

        # Get based on valid ID and Admin API
        id = "677fbd106592cb0065315618"
        post = Post(self.adminClient, id=id).get()
        slug = "bora-reduzir-nosso-tempo-de-tela-em-2025"

        self.assertTrue(post.exists)
        self.assertEqual(post.id, id)
        self.assertEqual(post.slug, slug)

        # Get based on valid slug and Content API
        slug = "russos-trazem-russos-pra-o-brasil"
        post = Post(self.adminClient, slug=slug).get()
        id = "6733507528037900661b15a6"

        self.assertEqual(post.slug, slug)
        self.assertEqual(post.id, id)
        self.assertEqual(post.uuid, "452d4a71-e2dc-4958-9c2e-0788f843997d")
        self.assertEqual(
            post.title, "Russos vendem turismo de parto no Brasil via Telegram"
        )
        self.assertEqual(post.comment_id, "6733507528037900661b15a6")
        self.assertEqual(
            post.feature_image,
            "https://nucleo.jor.br/content/images/2025/01/bebes-russos-brasileiros2-1.png",
        )
        self.assertEqual(post.featured, True)
        self.assertEqual(
            post.feature_image_caption,
            '<span style="white-space: pre-wrap;">Arte: Rodolfo Almeida</span>',
        )
        self.assertIsNone(post.feature_image_alt)
        self.assertEqual(post.visibility, "public")
        self.assertEqual(len(post.excerpt), 241)
        self.assertEqual(
            post.url,
            "https://nucleo.jor.br/reportagem/2025-01-09-russos-trazem-russos-pra-o-brasil/",
        )
        self.assertEqual(post.reading_time, 4)
        self.assertIsNone(post.email_subject)


if __name__ == "__main__":
    main()
