from unittest import TestCase, main
from ghost import AdminAPIClient
from ghost.exceptions import BadRequestException, GhostException
from ghost.models import Tier
from faker import Faker
from datetime import datetime
from dotenv import load_dotenv
from os import getenv

from ghost.models.tier import TierType

load_dotenv()


class TestTier(TestCase):
    fake = Faker()
    client = AdminAPIClient(getenv("GHOST_API_TOKEN", ""), getenv("GHOST_URL", ""))

    def test_init(self):
        with self.assertRaises(GhostException):
            Tier(self.client)

        id = "1235"
        slug = "dfdfbdf"

        tier = Tier(self.client, id=id)
        self.assertFalse(tier.exists)
        self.assertEqual(tier.client, self.client)
        self.assertEqual(tier.id, id)
        self.assertIsNone(tier.slug, slug)

        tier = Tier(self.client, slug=slug)
        self.assertFalse(tier.exists)
        self.assertEqual(tier.client, self.client)
        self.assertIsNone(tier.id, id)
        self.assertEqual(tier.slug, slug)

        tier = Tier(self.client, id=id, slug=slug)
        self.assertFalse(tier.exists)
        self.assertEqual(tier.client, self.client)
        self.assertEqual(tier.id, id)
        self.assertEqual(tier.slug, slug)

    def test_get(self):
        id = "6294176b84b84878740ac7f8"
        slug = "apoio-jornalismo"

        # Try with invalid ID
        tier = Tier(self.client, id="0000000")
        with self.assertRaises(GhostException):
            tier.get()
        self.assertFalse(tier.exists)

        # Retry with valid slug
        tier.slug = slug
        tier.get()

        # Try getting tier by ID and slug
        tier_by_id = Tier(self.client, id=id).get()
        tier_by_slug = Tier(self.client, slug=slug).get()

        # All tiers should exist on Ghost
        self.assertTrue(tier.exists)
        self.assertTrue(tier_by_id.exists)
        self.assertTrue(tier_by_slug.exists)

        # All IDs should be equal
        self.assertEqual(tier.id, tier_by_id.id)
        self.assertEqual(tier.id, tier_by_slug.id)

        # All slugs should be equal
        self.assertEqual(tier.slug, tier_by_id.slug)
        self.assertEqual(tier.slug, tier_by_slug.slug)

        # ID and slug should be the provided ones
        self.assertEqual(tier.id, id)
        self.assertEqual(tier.slug, slug)

        # Other information should be fetched from API as well
        self.assertEqual(tier.name, "Apoie nosso jornalismo")
        self.assertTrue(tier.active)
        self.assertEqual(tier.welcome_page_url, "https://nucleo.jor.br/account/")
        self.assertEqual(tier.visibility, "public")
        self.assertEqual(tier.trial_days, 0)
        if tier.description:
            self.assertEqual(len(tier.description), 67)
        else:
            self.fail("Description is empty")
        self.assertEqual(tier.tier_type, TierType.PAID)
        self.assertEqual(tier.currency, "BRL")
        self.assertEqual(tier.monthly_price, 1000)
        self.assertEqual(tier.yearly_price, 11000)
        self.assertTrue(isinstance(tier.created_at, datetime))
        self.assertEqual(len(tier.benefits), 9)
        self.assertTrue(isinstance(tier.updated_at, datetime))

    def test_create(self):
        slug = name = self.fake.word()
        tier = Tier(self.client, slug=slug, name=name, active=False)

        # Cannot get tier before creation
        with self.assertRaises(GhostException):
            tier.get()

        # Cannot create without slug or name
        with self.assertRaises(GhostException):
            Tier(self.client).create()

        # Now try creating with valid slug and name
        tier.create()
        self.assertTrue(tier.exists)
        self.assertIsNotNone(tier.id)
        self.assertEqual(tier.name, name)
        self.assertEqual(tier.slug, slug)
        self.assertFalse(tier.active)

        # Creating should allow fetching
        tier.get()

        # Cannot create tier again
        with self.assertRaises(GhostException):
            tier.create()

        # Even if forced to
        tier.exists = False
        with self.assertRaises(BadRequestException):
            tier.create()
        tier.exists = True

    def test_update(self):
        slug = self.fake.word()
        tier = Tier(self.client, slug=slug)

        # Cannot update a tier that doesn't exist
        with self.assertRaises(GhostException):
            tier.update()

        # Even if forced to
        tier.exists = True
        with self.assertRaises(BadRequestException):
            tier.update()
        tier.exists = False

        # Get valid tier
        slug = "nd"
        tier = Tier(self.client, slug=slug).get()

        # Retrieve info
        id = tier.id

        # Update name
        old_name = tier.name
        tier.name = self.fake.word()
        tier.update()

        # Check if everything is ok
        self.assertEqual(tier.id, id)
        self.assertEqual(tier.slug, slug)
        self.assertNotEqual(tier.name, old_name)

        # Now try updating slug
        tier.slug = self.fake.word()
        tier.update()
        self.assertEqual(tier.id, id)
        self.assertNotEqual(tier.slug, slug)
        self.assertNotEqual(tier.name, old_name)

        # Revert tier data
        tier.slug = slug
        tier.name = old_name
        tier.update()

    def test_dict(self):
        id = "6294176b84b84878740ac7f8"
        slug = "apoio-jornalismo"
        tier = Tier(self.client, id=id, slug=slug)

        tier_dict = tier.to_dict()
        self.assertEqual(
            set(tier_dict.keys()),
            {
                "id",
                "name",
                "slug",
                "active",
                "welcome_page_url",
                "visibility",
                "trial_days",
                "description",
                "type",
                "currency",
                "monthly_price",
                "yearly_price",
                "benefits",
            },
        )
        self.assertEqual(tier_dict["id"], id)
        self.assertEqual(tier_dict["name"], "Apoie nosso jornalismo")
        self.assertEqual(tier_dict["slug"], slug)
        self.assertEqual(tier)

    def test_summarize(self):
        id = "63e9160fca774efb785b279f"
        slug = "test"
        tier = Tier(self.client, id=id, slug=slug)

        tier_summary = tier.summarize()
        self.assertEqual(set(tier_summary.keys()), {"id", "slug"})
        self.assertEqual(tier_summary["id"], id)
        self.assertEqual(tier_summary["slug"], slug)


if __name__ == "__main__":
    main()
