from unittest import TestCase, main
from ghost.filter import Operator, Query


class TestFilter(TestCase):
    def test_resolve(self):
        query = Query("name", "João Teste 123")
        self.assertEqual(query.resolve(), "name:'João Teste 123'")

    def test_and(self):
        query = Query("email", "teste+123@teste.com") * Query(
            "name",
            "João Teste",
            Operator.NOT,
        )
        self.assertEqual(
            query.resolve(), "email:'teste+123@teste.com'+name:-'João Teste'"
        )

    def test_or(self):
        query = Query("email", "teste+123@teste.com") + Query(
            "name", ["João Teste", "João Sem Nome"]
        )
        self.assertEqual(
            query.resolve(),
            "email:'teste+123@teste.com',name:['João Teste', 'João Sem Nome']",
        )

    def test_operations(self):
        query_not = Query("email", "teste@teste.com", Operator.NOT)
        self.assertEqual(query_not.resolve(), "email:-'teste@teste.com'")

        query_gt = Query("year", 2000, Operator.GREATER)
        self.assertEqual(query_gt.resolve(), "year:>2000")

        query_ge = Query("year", 2005, Operator.GREATER_EQUAL)
        self.assertEqual(query_ge.resolve(), "year:>=2005")

        query_lt = Query("year", 2024, Operator.LESS)
        self.assertEqual(query_lt.resolve(), "year:<2024")

        query_le = Query("year", 2020, Operator.LESS_EQUAL)
        self.assertEqual(query_le.resolve(), "year:<=2020")

        query_contains = Query("name", "João", Operator.CONTAINS)
        self.assertEqual(query_contains.resolve(), "name:~'João'")

        query_starts = Query("email", "teste", Operator.STARTS_WITH)
        self.assertEqual(query_starts.resolve(), "email:~^'teste'")

        query_ends = Query("email", ".com", Operator.ENDS_WITH)
        self.assertEqual(query_ends.resolve(), "email:~$'.com'")

        query_in = Query("age", list(range(20, 25)))
        self.assertEqual(query_in.resolve(), "age:[20, 21, 22, 23, 24]")

    def test_precedence(self):
        query_1 = Query("a", "b") * Query("c", "d") + Query("e", "f")
        self.assertEqual(query_1.resolve(), "a:'b'+c:'d',e:'f'")

        query_2 = Query("a", "b") + Query("c", "d") * Query("e", "f") * Query("g", "h")
        self.assertEqual(query_2.resolve(), "a:'b',c:'d'+e:'f'+g:'h'")

        query_3 = Query("a", "b") + Query("c", "d") * (
            Query("e", "f") * Query("g", "h")
        )
        self.assertEqual(query_3.resolve(), query_2.resolve())

        query_4 = (Query("a", "b") + Query("c", "d")) * Query("e", "f")
        self.assertEqual(query_4.resolve(), "(a:'b',c:'d')+e:'f'")

        query_5 = Query("a", "b") * (Query("c", "d") + Query("e", "f"))
        self.assertEqual(query_5.resolve(), "a:'b'+(c:'d',e:'f')")

        queryzilla = (
            (Query("a", "b") + Query("c", "d") + Query("e", "f"))
            * Query("g", "h", Operator.NOT)
            * Query("i", ["j", "k", "l"])
            + Query("m", "n")
            + Query("o", "p")
            + (
                Query("q", "r") * Query("s", "t")
                + (Query("u", "v") * (Query("w", "x") + Query("y", "z")))
            )
        )
        self.assertEqual(
            queryzilla.resolve(),
            "(a:'b',c:'d',e:'f')+g:-'h'+i:['j', 'k', 'l'],m:'n',o:'p',q:'r'+s:'t',u:'v'+(w:'x',y:'z')",
        )


if __name__ == "__main__":
    main()
