from typing import Optional
from datetime import datetime
from ghost.exceptions import BadRequestException, UnavailableAPIException
import requests
import jwt
import json


class APIClient:
    def __init__(self, api_token: str, base_url: str):
        self.api_token = api_token
        self.base_url = base_url

    def __str__(self):
        return f"{type(self).__name__} <base_url={self.base_url}, api_token={self.api_token[:2]}****{self.api_token[-1]}>"

    def _generate_url(self, endpoint: str) -> str:
        return ""

    def _generate_error_msg(self, error_str: bytes):
        error = json.loads(error_str)["errors"][0]
        return f"{error['type']} - {error['message']} - {error['context']}"

    def process_request(self, response: requests.Response, endpoint: str) -> dict:
        if response.status_code >= 500:
            raise UnavailableAPIException(response.status_code)

        if response.status_code >= 400:
            raise BadRequestException(
                endpoint,
                response.status_code,
                self._generate_error_msg(response.content),
            )

        try:
            return json.loads(response.content)
        except json.decoder.JSONDecodeError:
            return {}

    def get(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        return {}

    def post(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        return {}

    def put(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        return {}

    def delete(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        return {}


class AdminAPIClient(APIClient):
    def _generate_url(self, endpoint: str):
        return self.base_url + "/ghost/api/admin" + endpoint

    def _generate_request_headers(self):
        # JWT config
        id, secret = self.api_token.split(":", 2)
        iat = int(datetime.now().timestamp())
        exp = iat + 5 * 60  # 5 minutes after IAT
        jwt_header = {"alg": "HS256", "typ": "JWT", "kid": id}
        jwt_payload = {"iat": iat, "exp": exp, "aud": "/admin/"}
        jwt_token = jwt.encode(jwt_payload, bytes.fromhex(secret), headers=jwt_header)

        return {"Authorization": f"Ghost {jwt_token}"}

    def get(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        response = requests.get(
            self._generate_url(endpoint),
            headers=self._generate_request_headers(),
            json=payload,
            params=params,
        )

        return self.process_request(response, endpoint)

    def post(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        response = requests.post(
            self._generate_url(endpoint),
            headers=self._generate_request_headers(),
            json=payload,
            params=params,
        )

        return self.process_request(response, endpoint)

    def put(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        response = requests.put(
            self._generate_url(endpoint),
            headers=self._generate_request_headers(),
            json=payload,
            params=params,
        )

        return self.process_request(response, endpoint)

    def delete(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        response = requests.delete(
            self._generate_url(endpoint),
            headers=self._generate_request_headers(),
            json=payload,
            params=params,
        )

        return self.process_request(response, endpoint)

    def post_in_api(
        self,
        api: str,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        response = requests.post(
            self.base_url + api + endpoint,
            headers=self._generate_request_headers(),
            json=payload,
            params=params,
        )

        return self.process_request(response, endpoint)


class ContentAPIClient(APIClient):
    def _generate_url(self, endpoint: str) -> str:
        return self.base_url + "/ghost/api/content" + endpoint

    def get(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        if not params:
            params = dict()
        params["key"] = self.api_token

        response = requests.get(
            self._generate_url(endpoint),
            json=payload,
            params=params,
        )

        return self.process_request(response, endpoint)

    def post(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        if not params:
            params = dict()
        params["key"] = self.api_token

        response = requests.post(
            self._generate_url(endpoint),
            json=payload,
            params=params,
        )

        return self.process_request(response, endpoint)

    def put(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        if not params:
            params = dict()
        params["key"] = self.api_token

        response = requests.put(
            self._generate_url(endpoint),
            json=payload,
            params=params,
        )

        return self.process_request(response, endpoint)

    def delete(
        self,
        endpoint: str,
        payload: Optional[dict] = None,
        params: Optional[dict] = None,
    ) -> dict:
        if not params:
            params = dict()
        params["key"] = self.api_token

        response = requests.delete(
            self._generate_url(endpoint),
            json=payload,
            params=params,
        )

        return self.process_request(response, endpoint)


if __name__ == "__main__":
    from dotenv import load_dotenv
    from os import getenv

    load_dotenv()

    client = AdminAPIClient(getenv("GHOST_API_TOKEN", ""), getenv("GHOST_URL", ""))
    jwt_token = client._generate_request_headers()["Authorization"][6:]
    jwt_secret = client.api_token.split(":", 2)[1]

    print(jwt_token)
    print(jwt_secret)
    print(
        jwt.api_jwt.decode_complete(
            jwt_token,
            jwt_secret,
            algorithms=["HS256"],
            options={"verify_signature": False},
        )
    )

    response = client.post(
        "/members/", payload={"members": [{"email": "teste@voltdata.info"}]}
    )
