from ghost.models.base_model import BaseModel
from ghost.client import AdminAPIClient
from ghost.exceptions import GhostException, BadRequestException
from ghost.filter import FilterObject
from typing import Optional, Union, List
from datetime import datetime


class Member(BaseModel):
    printable = ["id", "email", "created_at", "note"]

    def __init__(
        self,
        client: AdminAPIClient,
        id: Optional[str] = None,
        name: Optional[str] = None,
        email: Optional[str] = None,
        labels: Optional[set] = set(),
        newsletters: Optional[set] = set(),
        tiers: Optional[set] = set(),
        note: Optional[str] = None,
        avatar_image: Optional[str] = None,
    ):
        super().__init__()
        self.client = client
        self.id = id
        self.name = name
        self.email = email
        self.labels = labels
        self.newsletters = newsletters
        self.tiers = tiers
        self.note = note
        self.avatar_image = avatar_image

    def get(self):
        if self.id is None and self.email is None:
            raise GhostException("Cannot retrieve member with no email or id.")

        try:
            response = self.client.get("/members/", params={"filter": f"id:{self.id}"})
            if not response or not response["members"]:
                raise BadRequestException("/members/", 404, "")
        except BadRequestException as e:
            if e.status_code == 404:
                response = self.client.get(
                    "/members/", params={"filter": f"email:{self.email}"}
                )
            else:
                raise

        if not response["members"]:
            raise GhostException(
                f"No member found with given ID ({self.id}) and email ({self.email})"
            )

        self._assign_from_data(response["members"][0])
        super().get()
        return self

    def create(self):
        super().create()

        member_data = {
            "email": self.email,
            "avatar_image": self.avatar_image,
            "labels": [{"id": lab[0]} for lab in self.labels] if self.labels else [],
            "newsletters": (
                [{"id": n[0]} for n in self.newsletters] if self.newsletters else []
            ),
            "note": self.note if self.note else "",
        }

        response = self.client.post("/members/", payload={"members": [member_data]})
        self._assign_from_data(response["members"][0])
        self.exists = True

        return self

    def update(self):
        super().update()

        member_data = dict()
        if self.email:
            member_data["email"] = self.email
        if self.avatar_image:
            member_data["avatar_image"] = self.avatar_image
        # if self.labels is not None:
        # member_data["labels"] = [{"id": lab[0]} for lab in self.labels]
        if self.newsletters is not None:
            member_data["newsletters"] = [{"id": n[0]} for n in self.newsletters]
        if self.note:
            member_data["note"] = self.note

        response = self.client.put(
            f"/members/{self.id}", payload={"members": [member_data]}
        )
        self._assign_from_data(response["members"][0])

        return self

    def add_newsletters(self, newsletters: Union[str, set, list]):
        super().update()

        news_list = None
        if isinstance(newsletters, str):
            news_list = set([newsletters])
        elif not isinstance(newsletters, list) and not isinstance(newsletters, set):
            raise GhostException(
                f"'newsletters' needs to be str, list or set, not {type(newsletters)}"
            )
        else:
            news_list = set(newsletters)

        curr_news = set()
        if self.newsletters:
            curr_news = set(n[0] for n in self.newsletters)
        news_final = curr_news.union(news_list)

        payload = {"members": [{"newsletters": [{"id": n} for n in news_final]}]}
        response = self.client.put(f"/members/{self.id}", payload=payload)
        self._assign_from_data(response["members"][0])

        return self

    def remove_newsletters(self, newsletter_ids: Union[str, set, list]):
        super().update()

        news_list = None
        if isinstance(newsletter_ids, str):
            news_list = set([newsletter_ids])
        elif not isinstance(newsletter_ids, list) and not isinstance(
            newsletter_ids, set
        ):
            raise GhostException(
                f"'newsletter_ids' needs to be str, list or set, not {type(newsletter_ids)}"
            )
        else:
            news_list = set(newsletter_ids)

        curr_news = set()
        if self.newsletters:
            curr_news = set(n[0] for n in self.newsletters)
        news_final = curr_news - news_list

        payload = {"members": [{"newsletters": [{"id": n} for n in news_final]}]}
        response = self.client.put(f"/members/{self.id}", payload=payload)
        self._assign_from_data(response["members"][0])

        return self

    def add_tiers(self, tier_ids: Union[str, set, list]):
        super().update()

        tier_list = None
        if isinstance(tier_ids, str):
            tier_list = set([tier_ids])
        elif not isinstance(tier_ids, list) and not isinstance(tier_ids, set):
            raise GhostException(
                f"'tier_ids' needs to be str, list or set, not {type(tier_ids)}"
            )
        else:
            tier_list = set(tier_ids)

        curr_tiers = set()
        if self.tiers:
            curr_tiers = set(t[0] for t in self.tiers)
        tiers_final = curr_tiers.union(tier_list)

        payload = {"members": [{"tiers": [{"id": t} for t in tiers_final]}]}
        response = self.client.put(f"/members/{self.id}", payload=payload)
        self._assign_from_data(response["members"][0])

        return self

    def remove_tiers(self, tier_ids: Union[str, set, list]):
        super().update()

        tier_list = None
        if isinstance(tier_ids, str):
            tier_list = set([tier_ids])
        elif not isinstance(tier_ids, list) and not isinstance(tier_ids, set):
            raise GhostException(
                f"'tier_ids' needs to be str, list or set, not {type(tier_ids)}"
            )
        else:
            tier_list = set(tier_ids)

        curr_tiers = set()
        if self.tiers:
            curr_tiers = set(t[0] for t in self.tiers)
        tiers_final = curr_tiers - tier_list

        payload = {"members": [{"tiers": [{"id": t} for t in tiers_final]}]}
        response = self.client.put(f"/members/{self.id}", payload=payload)
        self._assign_from_data(response["members"][0])

        return self

    def send_magic_link(self):
        if not self.exists:
            raise GhostException("Cannot send magic link to member that doesn't exist")

        payload = {
            "email": self.email,
            "emailType": "signin",
            "integrityToken": self.client.api_token,
        }

        self.client.post_in_api("/members/api", "/send-magic-link", payload=payload)

        return self

    @classmethod
    def from_data(cls, client: AdminAPIClient, data: dict):
        if not data.get("member") or not data["member"].get("current"):
            raise GhostException("Cannot assign from incomplete data")

        member = Member(client)
        member.exists = True
        member._assign_from_data(data["member"]["current"])

        return member

    @classmethod
    def all(
        cls,
        client: AdminAPIClient,
        pagesize: Optional[int] = None,
        filters: Optional[dict | FilterObject] = {},
        limit: Optional[int] = None,
    ) -> List["Member"]:
        filter_str = ""
        if filters:
            if isinstance(filters, FilterObject):
                filter_str = filters.resolve()
            else:
                filter_str = "+".join([f"{k}:{v}" for k, v in filters.items()])

        params = dict()
        if filter_str != "":
            params["filter"] = filter_str

        if pagesize is not None:
            params["limit"] = pagesize
        else:
            params["limit"] = "all"

        data = client.get("/members/", params=params)

        members = data["members"]

        next = data["meta"]["pagination"]["next"]

        def loop_condition(next, limit, members):
            if limit:
                return bool(next) and len(members) < limit
            return bool(next)

        while loop_condition(next, limit, members):
            params["page"] = next
            data = client.get("/members/", params=params)
            members.extend(data["members"])
            next = data["meta"]["pagination"]["next"]

        return [Member.from_data(client, {"member": {"current": m}}) for m in members]

    def _assign_from_data(self, data: dict):
        self.id = data.get("id")
        self.email = data.get("email")
        self.name = data.get("name")
        self.avatar_image = data.get("avatar_image")
        self.note = data.get("note")
        self.status = data.get("status")
        self.created_at = datetime.fromisoformat(str(data.get("created_at")))
        self.updated_at = datetime.fromisoformat(str(data.get("updated_at")))
        self.labels = self._get_labels(data)
        self.newsletters = self._get_newsletters(data)
        self.tiers = self._get_tiers(data)

    def _get_newsletters(self, data: dict):
        newsletters = data.get("newsletters")
        if not newsletters:
            return set()

        return set((n["id"], n["name"]) for n in newsletters)

    def _get_labels(self, data: dict):
        labels = data.get("labels")
        if not labels:
            return set()

        return set((lab["id"], lab["slug"]) for lab in labels)

    def _get_tiers(self, data: dict):
        tiers = data.get("tiers")
        if not tiers or len(tiers) == 0:
            subscriptions = data.get("subscriptions")
            if not subscriptions or len(subscriptions) == 0:
                return set()
            tiers = [s["tier"] for s in subscriptions if s["status"] == "active"]

        if not tiers or len(tiers) == 0:
            return set()

        return set((t["id"], t["name"]) for t in tiers)
