from unittest import TestCase, main
from ghost import AdminAPIClient
from ghost.exceptions import GhostException, BadRequestException
from faker import Faker
from dotenv import load_dotenv
from os import getenv
from datetime import datetime, timedelta
import jwt


load_dotenv()


class TestClient(TestCase):
    def __init__(self, arg):
        super().__init__(arg)
        self.fake = Faker()

    def test_init(self):
        client = AdminAPIClient(getenv("GHOST_API_TOKEN"), getenv("GHOST_URL"))

        self.assertTrue(getenv("GHOST_API_TOKEN"))
        self.assertTrue(getenv("GHOST_URL"))
        self.assertEqual(client.api_token, getenv("GHOST_API_TOKEN"))
        self.assertEqual(client.base_url, getenv("GHOST_URL"))

    def test_generate_request_headers(self):
        client = AdminAPIClient(getenv("GHOST_API_TOKEN"), getenv("GHOST_URL"))
        headers = client._generate_request_headers()

        self.assertTrue(headers["Authorization"])
        self.assertRegex(headers["Authorization"], r"Ghost \w.\w.\w")

        jwt_token = headers["Authorization"][6:]
        secret = client.api_token.split(":", 2)[1]
        jwt_decoded = jwt.api_jwt.decode_complete(
            jwt_token, secret, algorithms=["HS256"], options={"verify_signature": False}
        )

        jwt_headers = jwt_decoded.get("header")
        jwt_payload = jwt_decoded.get("payload")

        self.assertTrue(jwt_headers)
        self.assertTrue(jwt_payload)

        self.assertEqual(jwt_headers["alg"], "HS256")
        self.assertEqual(jwt_headers["kid"], getenv("GHOST_API_TOKEN").split(":")[0])
        self.assertEqual(jwt_headers["typ"], "JWT")

        self.assertLessEqual(jwt_payload["iat"], int(datetime.now().timestamp()))
        self.assertEqual(jwt_payload["exp"], jwt_payload["iat"] + 5 * 60)
        self.assertLess(jwt_payload["iat"], jwt_payload["exp"])
        self.assertEqual(jwt_payload["aud"], "/admin/")

    def test_get(self):
        client = AdminAPIClient(getenv("GHOST_API_TOKEN"), getenv("GHOST_URL"))

        # Test a 400 request
        with self.assertRaises(BadRequestException):
            client.get("/teste/")

        # TODO: test a 500 request (how to make the server fail?)

        # Test a valid request
        response = client.get("/members/", params={"include": "newsletters"})

        self.assertTrue(response)
        self.assertTrue(response["members"])
        self.assertEqual(len(response["members"]), 15)  # Standard amount of members
        for member in response["members"]:
            self.assertIsNotNone(member.get("newsletters"))

    def test_post(self):
        client = AdminAPIClient(getenv("GHOST_API_TOKEN"), getenv("GHOST_URL"))

        # Test a 400 request
        with self.assertRaises(BadRequestException):
            client.post("/teste/")

        # TODO: test a 500 request (how to make the server fail?)

        # Test a valid request - Webhooks are a safe endpoint for both POST and DELETE
        response = client.post(
            "/webhooks/",
            payload={
                "webhooks": [
                    {"event": "post.published", "target_url": "https://henrieger.dev"}
                ]
            },
        )

        self.assertTrue(response)
        self.assertTrue(response["webhooks"])
        self.assertEqual(len(response["webhooks"]), 1)

        webhook = response["webhooks"][0]

        self.assertTrue(webhook["id"])
        self.assertEqual(webhook["event"], "post.published")
        self.assertEqual(webhook["target_url"], "https://henrieger.dev")

        created_at = datetime.fromisoformat(webhook["created_at"])
        self.assertLessEqual(
            abs(created_at - datetime.now(created_at.tzinfo)), timedelta(seconds=2)
        )

        # Remove created asset
        client.delete(
            "/webhooks/" + webhook["id"],
        )

    def test_put(self):
        client = AdminAPIClient(getenv("GHOST_API_TOKEN"), getenv("GHOST_URL"))

        # Test a 400 request
        with self.assertRaises(BadRequestException):
            client.put("/teste/")

        # TODO: test a 500 request (how to make the server fail?)

        # Retrieve test client
        response = client.get(
            "/members/", params={"filter": "email:teste@voltdata.info"}
        )
        member = response["members"][0]

        # Test a valid request
        test_date = datetime.now().strftime("%d/%m/%Y %H:%M:%S")
        response = client.put(
            f"/members/{member['id']}/",
            payload={"members": [{"note": member["note"] + f"\nTeste {test_date}"}]},
        )

        self.assertTrue(response)
        self.assertTrue(response["members"])
        self.assertEqual(len(response["members"]), 1)

        member = response["members"][0]

        self.assertTrue(member["id"])
        self.assertEqual(member["email"], "teste@voltdata.info")
        self.assertTrue(member["note"].endswith(test_date))

    def test_delete(self):
        client = AdminAPIClient(getenv("GHOST_API_TOKEN"), getenv("GHOST_URL"))

        # Test a 400 request
        with self.assertRaises(BadRequestException):
            client.delete("/teste/")

        # TODO: test a 500 request (how to make the server fail?)

        # Test a valid request - Webhooks are a safe endpoint for both POST and DELETE
        response = client.post(
            "/webhooks/",
            payload={
                "webhooks": [
                    {"event": "post.published", "target_url": "https://henrieger.dev"}
                ]
            },
        )

        # Validation of post has its own tests
        webhook = response["webhooks"][0]

        # Remove created asset
        response = client.delete(
            "/webhooks/" + webhook["id"],
        )

        # Nothing should be returned
        self.assertIsNone(response)

        # Item shouldn't exist anymore, and updates should therefore fail
        with self.assertRaises(BadRequestException):
            client.put("/webhooks/", params={"filter": f"id:{webhook['id']}"})


if __name__ == "__main__":
    main()
