from unittest import TestCase, main
from ghost.webhook import webhook_authentication, MemberData
from fastapi import FastAPI, Depends, Response
from fastapi.testclient import TestClient
from dotenv import load_dotenv
from os import getenv
import hmac
import json


load_dotenv()

app = FastAPI()
client = TestClient(app)


@app.post("/", status_code=204)
def test_endpoint(
    input: MemberData,
    response: Response,
    request: dict = Depends(webhook_authentication),
):
    member_data = request["member"]
    return member_data


class TestWebhook(TestCase):
    def test_ghost_webhook_authentication(self):
        # Request without headers should fail with status 422
        response = client.post("/")
        self.assertEqual(response.status_code, 422)

        # Request with invalid headers should fail with status 403
        response = client.post("/", headers={"x-ghost-signature": ""})
        self.assertEqual(response.status_code, 403)

        # Request with valid format but incorrect credentials should fail with status 401
        response = client.post("/", headers={"x-ghost-signature": "sha256=abcd, t=0"})
        self.assertEqual(response.status_code, 401)

        # Request with valid headers but incorrect json body should fail with status 422
        payload = {}
        signature = hmac.new(
            key=getenv("GHOST_WEBHOOK_SECRET").encode(),
            msg=(json.dumps(payload) + "0").encode(),
            digestmod="sha256",
        ).hexdigest()
        response = client.post(
            "/", headers={"x-ghost-signature": f"sha256={signature}, t=0"}, json=payload
        )
        self.assertEqual(response.status_code, 422)

        # Valid request should succeed with status 204
        payload = {
            "member": {
                "current": {
                    "id": "test-id",
                    "email": "teste@voltdata.info",
                    "labels": [],
                },
                "previous": {"labels": [{"id": "test-label-id", "slug": "test-label"}]},
            }
        }
        signature = hmac.new(
            key=getenv("GHOST_WEBHOOK_SECRET").encode(),
            msg=json.dumps(payload).encode() + b"0",
            digestmod="sha256",
        ).hexdigest()
        response = client.post(
            "/", headers={"x-ghost-signature": f"sha256={signature}, t=0"}, json=payload
        )
        self.assertEqual(response.status_code, 204)


if __name__ == "__main__":
    main()
