# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['python_redis_orm', 'python_redis_orm.tests']

package_data = \
{'': ['*']}

install_requires = \
['redis>=3.5.3,<4.0.0']

setup_kwargs = {
    'name': 'python-redis-orm',
    'version': '0.1.7',
    'description': 'Python Redis ORM library turns redis to a fully functional in-memory database and speeds a developmet up, inspired by Django ORM',
    'long_description': '# redis-orm\n\n## **Python Redis ORM, turns redis to a fully functional in-memory database, inspired by Django ORM**\n\nFor one project, I needed to work with redis, but redis-py provides a minimum level of work with redis. I didn\'t find any Django-like ORM for redis, so I wrote this library, then there will be a port to Django.\n\n### Working with this library, you are expected:\n\n- Fully works in 2021\n- Django-like architecture\n- Easy adaptation to your needs\n- Adequate informational messages and error messages\n- Built-in RedisModel class, with:\n    - TTL (Time To Live), applies if no ttl on field\n- 6 built-in types of fields:\n    - RedisField - base class for nesting\n    - RedisString - string\n    - RedisNumber - int or float\n    - RedisId - instances IDs\n    - RedisDateTime - for work with date and time, via python datetime\n    - RedisForeignKey - for links to other redis models\n- All fields supports:\n    - Automatically serialization and deserialization\n    - TTL (Time To Live)\n    - Default values\n    - Providing functions to default values\n    - Allow null values setting\n    - Choices\n- Extras:\n    - Ignore deserialization errors setting - do not raise errors, while deserealizing data\n    - Save consistency setting - show structure-first data\n\n\n# Installation\n`pip install python-redis-orm`\n\n[Here is PyPI](https://pypi.org/project/python-redis-orm/)\n\nObviously, you need to install and run redis server on your machine, we support v3+ \n\n\n# Usage\n\nAll features:\n[full_test.py](https://github.com/gh0st-work/python_redis_orm/blob/master/python_redis_orm/tests/full_test.py)\n```python\nimport datetime\nimport random\nfrom time import sleep\nfrom python_redis_orm.core import *\n\n\ndef generate_token(chars_count):\n    allowed_chars = \'QWERTYUIOPASDFGHJKLZXCVBNM1234567890\'\n    token = f\'{"".join([random.choice(allowed_chars) for i in range(chars_count)])}\'\n    return token\n\n\ndef generate_token_12_chars():\n    return generate_token(12)\n\n\nclass BotSession(RedisModel):\n    session_token = RedisString(default=generate_token_12_chars)\n    created = RedisDateTime(default=datetime.datetime.now)\n\n\nclass TaskChallenge(RedisModel):\n    bot_session = RedisForeignKey(model=BotSession)\n    task_id = RedisNumber(default=0, null=False)\n    status = RedisString(default=\'in_work\', choices={\n        \'in_work\': \'В работе\',\n        \'completed\': \'Завершён успешно\',\n        \'completed_frozen_points\': \'Завершён успешно, получил поинты в холде\',\n        \'completed_points\': \'Завершён успешно, получил поинты\',\n        \'completed_decommissioning\': \'Завершён успешно, поинты списаны\',\n        \'failed_bot\': \'Зафейлил бот\',\n        \'failed_task_creator\': \'Зафейлил создатель задания\',\n    }, null=False)\n    account_checks_count = RedisNumber(default=0)\n    created = RedisDateTime(default=datetime.datetime.now)\n\n\nclass TtlCheckModel(RedisModel):\n    redis_number_with_ttl = RedisNumber(default=0, null=False, ttl=5)\n\n\nclass MetaTtlCheckModel(RedisModel):\n    redis_number = RedisNumber(default=0, null=False)\n\n    class Meta:\n        ttl = 5\n\n\n# class DjangoForeignKeyModel(RedisModel):\n#     foreign_key = RedisDjangoForeignKey(model=Proxy)\n\n\ndef get_redis_instance():\n    REDIS_HOST = \'localhost\'\n    REDIS_PORT = 6379\n    redis_instance = redis.StrictRedis(\n        decode_responses=True,\n        host=REDIS_HOST,\n        port=REDIS_PORT,\n        db=0,\n    )\n    return redis_instance\n\n\ndef clean_db_after_test(redis_instance, prefix):\n    for key in redis_instance.scan_iter(f\'{prefix}:*\'):\n        redis_instance.delete(key)\n\n\ndef basic_test(redis_instance, prefix):\n    try:\n        redis_root = RedisRoot(\n            prefix=prefix,\n            redis_instance=redis_instance,\n            ignore_deserialization_errors=True\n        )\n        redis_root.register_models([\n            TaskChallenge,\n        ])\n        for i in range(5):\n            TaskChallenge(\n                redis_root=redis_root,\n                status=\'in_work\',\n            ).save()\n        task_challenges_without_keys = redis_root.get(TaskChallenge)\n        task_challenges_with_keys = redis_root.get(TaskChallenge, return_dict=True)\n        have_exception = False\n        if not len(task_challenges_without_keys):\n            have_exception = True\n        if not task_challenges_with_keys:\n            have_exception = True\n        else:\n            if not task_challenges_with_keys.keys():\n                have_exception = True\n            else:\n                if len(list(task_challenges_with_keys.keys())) != len(task_challenges_without_keys):\n                    have_exception = True\n    except BaseException as ex:\n        have_exception = True\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef auto_reg_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True\n    )\n    task_challenge_1 = TaskChallenge(\n        redis_root=redis_root,\n        status=\'in_work\',\n    ).save()\n    try:\n        task_challenges = redis_root.get(TaskChallenge)\n        have_exception = False\n    except BaseException as ex:\n        have_exception = True\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef no_redis_instance_test(*args, **kwargs):\n    try:\n        redis_root = RedisRoot(\n            ignore_deserialization_errors=True\n        )\n        task_challenge_1 = TaskChallenge(\n            redis_root=redis_root,\n            status=\'in_work\',\n        )\n        task_challenge_1.save()\n        task_challenges = redis_root.get(TaskChallenge)\n        have_exception = False\n        clean_db_after_test(redis_root.redis_instance, redis_root.prefix)\n    except BaseException as ex:\n        have_exception = True\n    return have_exception\n\n\ndef choices_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True\n    )\n    task_challenge_1 = TaskChallenge(\n        redis_root=redis_root,\n        status=\'bruh\',\n    )\n    try:\n        task_challenge_1.save()\n        task_challenges = redis_root.get(TaskChallenge)\n        have_exception = True\n    except BaseException as ex:\n        have_exception = False\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef order_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True\n    )\n    for i in range(3):\n        TaskChallenge(\n            redis_root=redis_root\n        ).save()\n    have_exception = True\n    try:\n        task_challenges = redis_root.get(TaskChallenge)\n        first_task_challenge = redis_root.order(task_challenges, \'id\')[0]\n        last_task_challenge = redis_root.order(task_challenges, \'-id\')[0]\n        if first_task_challenge[\'id\'] == 1 and last_task_challenge[\'id\'] == len(task_challenges):\n            have_exception = False\n    except BaseException as ex:\n        pass\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef filter_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True\n    )\n    have_exception = True\n    try:\n        same_tokens_count = 2\n        random_tokens_count = 8\n        same_token = generate_token(50)\n        random_tokens = [generate_token(50) for i in range(random_tokens_count)]\n        for i in range(same_tokens_count):\n            BotSession(redis_root, session_token=same_token).save()\n        for random_token in random_tokens:\n            BotSession(redis_root, session_token=random_token).save()\n        task_challenges_with_same_token = redis_root.get(BotSession, session_token=same_token)\n        if len(task_challenges_with_same_token) == same_tokens_count:\n            have_exception = False\n    except BaseException as ex:\n        print(ex)\n\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef update_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True\n    )\n    have_exception = True\n    try:\n        bot_session_1 = BotSession(redis_root, session_token=\'123\').save()\n        bot_session_1_id = bot_session_1[\'id\']\n        redis_root.update(BotSession, bot_session_1, session_token=\'234\')\n        bot_sessions_filtered = redis_root.get(BotSession, id=bot_session_1_id)\n        if len(bot_sessions_filtered) == 1:\n            bot_session_1_new = bot_sessions_filtered[0]\n            if \'session_token\' in bot_session_1_new.keys():\n                if bot_session_1_new[\'session_token\'] == \'234\':\n                    have_exception = False\n    except BaseException as ex:\n        print(ex)\n\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef functions_like_defaults_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True\n    )\n    have_exception = False\n    try:\n        bot_session_1 = BotSession(redis_root).save()\n        bot_session_2 = BotSession(redis_root).save()\n        if bot_session_1.session_token == bot_session_2.session_token:\n            have_exception = True\n    except BaseException as ex:\n        pass\n\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef redis_foreign_key_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True\n    )\n    have_exception = True\n    try:\n        bot_session_1 = BotSession(\n            redis_root=redis_root,\n        ).save()\n        task_challenge_1 = TaskChallenge(\n            redis_root=redis_root,\n            bot_session=bot_session_1\n        ).save()\n        bot_sessions = redis_root.get(BotSession)\n        bot_session = redis_root.order(bot_sessions, \'-id\')[0]\n        task_challenges = redis_root.get(TaskChallenge)\n        task_challenge = redis_root.order(task_challenges, \'-id\')[0]\n        if type(task_challenge[\'bot_session\']) == dict:\n            if task_challenge[\'bot_session\'] == bot_session:\n                have_exception = False\n    except BaseException as ex:\n        print(ex)\n\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\n# def django_foreign_key_test(redis_instance, prefix):\n#     redis_root = RedisRoot(\n#         prefix=prefix,\n#         redis_instance=redis_instance,\n#         ignore_deserialization_errors=True\n#     )\n#     have_exception = True\n#     try:\n#         proxy = Proxy.objects.all()[0]\n#         django_foreign_key_model = DjangoForeignKeyModel(\n#             redis_root=redis_root,\n#             foreign_key=proxy,\n#         ).save()\n#         django_foreign_key_models = redis_root.get(DjangoForeignKeyModel)\n#         django_foreign_key_model = redis_root.order(django_foreign_key_models, \'-id\')[0]\n#         if django_foreign_key_model[\'foreign_key\'] == proxy:\n#             have_exception = False\n#     except BaseException as ex:\n#         print(ex)\n#     clean_db_after_test(redis_instance, prefix)\n#     return have_exception\n\n\ndef delete_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True\n    )\n    have_exception = True\n    try:\n        bot_session_1 = BotSession(\n            redis_root=redis_root,\n        ).save()\n        task_challenge_1 = TaskChallenge(\n            redis_root=redis_root,\n            bot_session=bot_session_1\n        ).save()\n        redis_root.delete(BotSession, bot_session_1)\n        redis_root.delete(TaskChallenge, task_challenge_1)\n        bot_sessions = redis_root.get(BotSession)\n        task_challenges = redis_root.get(TaskChallenge)\n        if len(bot_sessions) == 0 and len(task_challenges) == 0:\n            have_exception = False\n    except BaseException as ex:\n        print(ex)\n\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef ttl_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True\n    )\n    have_exception = True\n    try:\n        ttl_check_model_1 = TtlCheckModel(\n            redis_root=redis_root,\n        ).save()\n        ttl_check_models = redis_root.get(TtlCheckModel)\n        if len(ttl_check_models):\n            ttl_check_model = ttl_check_models[0]\n            if \'redis_number_with_ttl\' in ttl_check_model.keys():\n                sleep(6)\n                ttl_check_models = redis_root.get(TtlCheckModel)\n                if len(ttl_check_models):\n                    ttl_check_model = ttl_check_models[0]\n                    if \'redis_number_with_ttl\' not in ttl_check_model.keys():\n                        have_exception = False\n    except BaseException as ex:\n        print(ex)\n\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef save_consistency_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True,\n        save_consistency=True,\n    )\n    have_exception = True\n    try:\n        ttl_check_model_1 = TtlCheckModel(\n            redis_root=redis_root,\n        ).save()\n        ttl_check_models = redis_root.get(TtlCheckModel)\n        if len(ttl_check_models):\n            ttl_check_model = ttl_check_models[0]\n            if \'redis_number_with_ttl\' in ttl_check_model.keys():\n                sleep(6)\n                ttl_check_models = redis_root.get(TtlCheckModel)\n                if len(ttl_check_models):\n                    ttl_check_model = ttl_check_models[0]\n                    if \'redis_number_with_ttl\' in ttl_check_model.keys():  # because consistency is saved\n                        have_exception = False\n    except BaseException as ex:\n        print(ex)\n\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef meta_ttl_test(redis_instance, prefix):\n    redis_root = RedisRoot(\n        prefix=prefix,\n        redis_instance=redis_instance,\n        ignore_deserialization_errors=True,\n    )\n    have_exception = True\n    try:\n        meta_ttl_check_model_1 = MetaTtlCheckModel(\n            redis_root=redis_root,\n        ).save()\n        meta_ttl_check_models = redis_root.get(MetaTtlCheckModel)\n        if len(meta_ttl_check_models):\n            meta_ttl_check_model = meta_ttl_check_models[0]\n            if \'redis_number\' in meta_ttl_check_model.keys():\n                sleep(6)\n                meta_ttl_check_models = redis_root.get(MetaTtlCheckModel)\n                if not len(meta_ttl_check_models):\n                    have_exception = False\n    except BaseException as ex:\n        print(ex)\n\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef economy_test(redis_instance, prefix):\n    have_exception = True\n    try:\n\n        redis_root = RedisRoot(\n            prefix=prefix,\n            redis_instance=redis_instance,\n            ignore_deserialization_errors=True,\n            economy=True\n        )\n        started_in_economy = datetime.datetime.now()\n        for i in range(10):\n            task_challenge_1 = TaskChallenge(\n                redis_root=redis_root,\n                status=\'in_work\',\n            ).save()\n            redis_root.update(TaskChallenge, task_challenge_1, account_checks_count=1)\n        ended_in_economy = datetime.datetime.now()\n        economy_time = (ended_in_economy - started_in_economy).total_seconds()\n        clean_db_after_test(redis_instance, prefix)\n\n        redis_root = RedisRoot(\n            prefix=prefix,\n            redis_instance=redis_instance,\n            ignore_deserialization_errors=True,\n            economy=False\n        )\n        started_in_no_economy = datetime.datetime.now()\n        for i in range(10):\n            task_challenge_1 = TaskChallenge(\n                redis_root=redis_root,\n                status=\'in_work\',\n            ).save()\n            redis_root.update(TaskChallenge, task_challenge_1, account_checks_count=1)\n        ended_in_no_economy = datetime.datetime.now()\n        no_economy_time = (ended_in_no_economy - started_in_no_economy).total_seconds()\n        clean_db_after_test(redis_instance, prefix)\n        economy_percent = round((no_economy_time / economy_time - 1) * 100, 2)\n        economy_symbol = (\'+\' if economy_percent > 0 else \'-\')\n        print(f\'Economy gives {economy_symbol}{economy_percent}% efficiency\')\n        if economy_symbol == \'+\':\n            have_exception = False\n    except BaseException as ex:\n        print(ex)\n\n    clean_db_after_test(redis_instance, prefix)\n    return have_exception\n\n\ndef run_tests():\n    redis_instance = get_redis_instance()\n    tests = [\n        basic_test,\n        auto_reg_test,\n        no_redis_instance_test,\n        choices_test,\n        order_test,\n        filter_test,\n        functions_like_defaults_test,\n        redis_foreign_key_test,\n        # django_foreign_key_test,\n        update_test,\n        delete_test,\n        ttl_test,\n        save_consistency_test,\n        meta_ttl_test,\n        economy_test,\n    ]\n    results = []\n    started_in = datetime.datetime.now()\n    print(\'STARTING TESTS\\n\')\n    for i, test in enumerate(tests):\n        print(f\'Starting {int(i + 1)} test: {test.__name__.replace("_", " ")}\')\n        test_started_in = datetime.datetime.now()\n        result = not test(redis_instance, test.__name__)\n        test_ended_in = datetime.datetime.now()\n        test_time = (test_ended_in - test_started_in).total_seconds()\n        print(f\'{result = } / {test_time}s\\n\')\n        results.append(result)\n    ended_in = datetime.datetime.now()\n    time = (ended_in - started_in).total_seconds()\n    success_message = \'SUCCESS\' if all(results) else \'FAILED\'\n    print(\'\\n\'\n          f\'{success_message}!\\n\')\n    results_success_count = 0\n    for i, result in enumerate(results):\n        result_message = \'SUCCESS\' if result else \'FAILED\'\n        print(f\'Test {(i + 1)}/{len(results)}: {result_message} ({tests[i].__name__.replace("_", " ")})\')\n        if result:\n            results_success_count += 1\n    print(f\'\\n\'\n          f\'{results_success_count} / {len(results)} tests ran successfully\\n\'\n          f\'All tests completed in {time}s\\n\')\n\n\nif __name__ == \'__main__\':\n    run_tests()\n\n```',
    'author': 'Anton Nechaev',
    'author_email': 'antonnechaev990@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/gh0st-work/python_redis_orm',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
